unit DDirectDraw;

interface

uses sysutils, DDraw, Dvp, D3D, DirectX, Windows, Classes, Controls,
{$IFDEF VER90}
  Dialogs,
{$ENDIF}
  Forms;

type
  EDelphiDirectDrawError = class(Exception);
  EDirectDrawError = class(EDirectXError);

  {* Enumeration records *}
  TDisplayModeEnum = record
    SurfaceDesc: TDDSurfaceDesc;
  end;
  PDisplayModeEnum = ^TDisplayModeEnum;

  TDirectDrawEnum = record
    GUID: TGUID;
    DriverDescription: String[100];
    DriverName: String[100];
  end;
  PDirectDrawEnum = ^TDirectDrawEnum;


  {*  *}

  TDirectDrawClipper = class;
  TDirectDrawSurface = class;
  TDirectDrawPalette = class;
  TDirectDrawVideoPort = class;

  TDirectDraw = class( TObject )
  private
    FInterface1: IDirectDraw;
    FClippers, FPalettes, FSurfaces: TList;
    FFourCCCodes: PDWord;
    FFourCCCodeCount: DWord;

    function GetClipper( Index: Integer ): TDirectDrawClipper;
    function GetClipperCount: Integer;
    function GetDisplayMode: TDDSurfaceDesc;
    function GetDriverCaps: TDDCaps;
    function GetFourCCCodes( Index: Integer ): Longint;
    function GetFourCCCodeCount: Integer;
    procedure InitFourCCCodes;
    function GetGDISurface: TDirectDrawSurface;
    function GetHELCaps: TDDCaps;
    function GetIsVerticalBlank: Boolean;
    function GetMonitorFrequency: DWord;
    function GetPalette( Index: Integer ): TDirectDrawPalette;
    function GetPaletteCount: Integer;
    function GetScanLine: Longint;
    function GetSurface( Index: Integer ): TDirectDrawSurface;
    function GetSurfaceCount: Integer;

  protected
    FCurInterface: IUnknown;

    procedure InitializeInterface( lpGUID: PGUID ); virtual;
    procedure DestroyInterface; virtual;

  public
    constructor Create( lpGUID: PGUID ); virtual;
    destructor Destroy; override ;

    procedure EnumerateDisplayModes( Flags: DWord;
        pSurfaceDesc: PDDSurfaceDesc; List: TList );
    procedure EnumerateSurfaces( Flags: Longint;
      const SurfaceDesc: TDDSurfaceDesc; List: TList );
    procedure FlipToGDISurface;
    procedure Notification( Obj: TObject; Operation: TOperation ); virtual;
    procedure RestoreDisplayMode;
    procedure SetCooperativeLevel( Window: HWND; Flags: Longint );
    procedure SetDisplayMode( Width: Longint; Height: Longint;
        Bpp: Longint );
    function WaitForVerticalBlank( Flags: Longint ): Boolean;

    function LoadBitmap( const Name: String; Width, Height: Integer ):
      TDirectDrawSurface;
    function LoadPalette( const Name: String ): TDirectDrawPalette;

    property Clippers[Index: Integer]: TDirectDrawClipper read GetClipper;
    property ClipperCount: Integer read GetClipperCount;
    property DisplayMode: TDDSurfaceDesc read GetDisplayMode;
    property DriverCaps: TDDCaps read GetDriverCaps;
    property FourCCCodes[Index: Integer]: Longint read GetFourCCCodes;
    property FourCCCodeCount: Integer read GetFourCCCodeCount;
    property GDISurface: TDirectDrawSurface read GetGDISurface;
    property HELCaps: TDDCaps read GetHELCaps;
    property Interface1: IDirectDraw read FInterface1;
    property IsVerticalBlank: Boolean read GetIsVerticalBlank;
    property MonitorFrequency: DWord read GetMonitorFrequency;
    property Palettes[Index: Integer]: TDirectDrawPalette read GetPalette;
    property PaletteCount: Integer read GetPaletteCount;
    property ScanLine: Longint read GetScanLine;
    property Surfaces[Index: Integer]: TDirectDrawSurface read GetSurface;
    property SurfaceCount: Integer read GetSurfaceCount;
  end;


  TDirectDraw2 = class( TDirectDraw )
  private
    FVideoPorts: TList;
    FInterface2: IDirectDraw2;
    FDirect3D: IDirect3D;
    FDDVideoPortContainer: IDDVideoPortContainer;

    function GetVideoPortStatus( PortId: Longint ): TDDVideoPortStatus;
    function GetVideoPort( Index: Integer ): TDirectDrawVideoPort;
    function GetVideoPortCount: Integer;

  protected
    procedure InitializeInterface( lpGUID: PGUID ); override;
    procedure DestroyInterface; override;

  public
    constructor Create( lpGUID: PGUID ); override;
    destructor Destroy; override ;

    procedure GetAvailableVidMem( var Caps: TDDSCaps;
      var Total, Free: DWord );
    procedure SetDisplayMode( Width: Longint; Height: Longint;
        Bpp: Longint; RefreshRate: Longint; Flags: Longint );
    procedure EnumerateVideoPorts( Flags: Longint;
      VideoPortCaps: PDDVideoPortCaps; List: TList );
    procedure GetVideoPortConnectInfo( PortId: DWord; var NumEntries: DWord;
      pConnectInfo: PDDVideoPortConnect );
    procedure Notification(Obj: TObject; Operation: TOperation); override;

    property DDVideoPortContainer: IDDVideoPortContainer
      read FDDVideoPortContainer;
    property Direct3D: IDirect3D read FDirect3D;
    property Interface2: IDirectDraw2 read FInterface2;
    property VideoPorts[Index: Integer]: TDirectDrawVideoPort read GetVideoPort;
    property VideoPortCount: Integer read GetVideoPortCount;
    property VideoPortStatus[PortId: Longint]: TDDVideoPortStatus
      read GetVideoPortStatus;
  end;


  TDirectDrawSurface = class( TObject )
  private
    FOwner: TDirectDraw;
    FInterface1: IDirectDrawSurface;
    FGammaControl: IDirectDrawGammaControl;
    FChain: TList;

    function GetCaps: TDDSCaps;
    function GetClipper: TDirectDrawClipper;
    function GetOverlayPosition: TPoint;
    function GetPalette: TDirectDrawPalette;
    function GetPixelFormat: TDDPixelFormat;
    function GetSurfaceDesc: TDDSurfaceDesc;
    function GetIsLost: Boolean;

    procedure SetClipper( DDClipper: TDirectDrawClipper );
    procedure SetOverlayPosition( const Pos: TPoint );
    procedure SetPalette( DDPalette: TDirectDrawPalette );

    constructor CreateChained( Root: TDirectDrawSurface;
      Interf: IDirectDrawSurface );

  protected
    FCurInterface: IUnknown;

    procedure InitializeInterface; virtual;
    procedure DestroyInterface; virtual;

  public
    constructor Create( Owner: TDirectDraw; const SurfaceDesc: TDDSurfaceDesc );
    constructor CreateDuplicated( Surface: TDirectDrawSurface );
    destructor Destroy; override;

    procedure AddAttachedSurface( AttachedSurface: TDirectDrawSurface );
    procedure AddOverlayDirtyRect( const Rect: TRect );
    procedure Blt( pDestRect: PRect;
        SrcSurface: TDirectDrawSurface; pSrcRect: PRect ;
        Flags: Longint; pBltFx: PDDBltFx );
    procedure BltBatch( const BltBatch: TDDBltBatch; Count: Longint;
        Flags: Longint );
    procedure BltFast( X: Longint; Y: Longint;
        SrcSurface: TDirectDrawSurface; pSrcRect: PRect;
        Trans: Longint );
    procedure DeleteAttachedSurface( Flags: Longint ;
        AttachedSurface: TDirectDrawSurface);
    procedure EnumerateAttachedSurfaces ( List: TList );
    procedure EnumerateOverlayZOrders( Flags: Longint; List: TList );
    procedure Flip( SurfaceTargetOverride: TDirectDrawSurface ;
        Flags: Longint );
    function GetAttachedSurface( Caps: TDDSCaps ): TDirectDrawSurface;
    function GetBltStatus( Flags: Longint ):  Boolean;
    function GetColorKey( Flags: Longint ): TDDColorKey;
    function GetDC( var DeviceContext: hDC ): Boolean;
//    procedure GetGammaRamp( Flags: DWord; var RampData: TGammaRamp );
    function GetFlipStatus( Flags: Longint ): Boolean;
    procedure Lock( pDestRect: PRect; var SurfaceDesc: TDDSurfaceDesc;
      Flags: Longint; hEvent: THandle );
    procedure ReleaseDC( const DeviceContext: HDC );
    function Restore: Boolean;
    procedure SetColorKey( Flags: Longint; const ColorKey: TDDColorKey );
//    procedure SetGammaRamp( Flags: DWord; const RampData: TGammaRamp );
    function Unlock: Pointer;
    procedure UpdateOverlay( const SrcRect: TRect;
        DestSurface: TDirectDrawSurface; const DestRect: TRect;
        Flags: Longint; const OverlayFx: TDDOverlayFx );
    procedure UpdateOverlayDisplay( Flags: Longint );
    procedure UpdateOverlayZOrder( Flags: Longint;
      SurfaceReference: TDirectDrawSurface );

    function ColorMatchRGB( rgb: COLORREF ): Longint;
    procedure CopyBitmap( BitmapHandle: HBITMAP; Left, Top, Width,
      Height: Integer);
    procedure ReloadBitmap( const Name: String );
    procedure SetRGBColorKey( rgb: COLORREF );

    property Caps: TDDSCaps read GetCaps;
    property Clipper: TDirectDrawClipper read GetClipper write SetClipper;
    property GammaControl: IDirectDrawGammaControl read FGammaControl;
    property Interface1: IDirectDrawSurface read FInterface1;
    property OverlayPosition: TPoint read GetOverlayPosition
      write SetOverlayPosition;
    property Owner: TDirectDraw read FOwner;
    property Palette: TDirectDrawPalette read GetPalette write SetPalette;
    property PixelFormat: TDDPixelFormat read GetPixelFormat;
    property SurfaceDesc: TDDSurfaceDesc read GetSurfaceDesc;
    property IsLost: Boolean read GetIsLost;
  end;


  TDirectDrawSurface2 = class( TDirectDrawSurface )
  private
    FInterface2: IDirectDrawSurface2;
    FDirect3DTexture: IDirect3DTexture;

  protected
    procedure InitializeInterface; override;
    procedure DestroyInterface; override;

  public
    procedure AddAttachedSurface( AttachedSurface: TDirectDrawSurface2 );
    procedure PageLock( Flags: Longint );
    procedure PageUnlock( Flags: Longint );

    property Direct3DTexture: IDirect3DTexture read FDirect3DTexture;
    property Interface2: IDirectDrawSurface2 read FInterface2;
  end;


  TDirectDrawSurface3 = class( TDirectDrawSurface2 )
  private
    FInterface3: IDirectDrawSurface3;

    function GetSurfaceDesc: TDDSurfaceDesc;
    procedure SetSurfaceDesc( const SurfaceDesc: TDDSurfaceDesc );

  protected
    procedure InitializeInterface; override;
    procedure DestroyInterface; override;

  public
    procedure AddAttachedSurface( AttachedSurface: TDirectDrawSurface3 );

    property Interface3: IDirectDrawSurface3 read FInterface3;
    property SurfaceDesc: TDDSurfaceDesc read GetSurfaceDesc
      write SetSurfaceDesc;
  end;


  TDirectDrawPalette = class( TObject )
  private
    FOwner: TDirectDraw;
    FInterface1: IDirectDrawPalette;

  protected
    procedure InitializeInterface( Flags: Longint; lpColorTable: PPaletteEntry ); virtual;
    procedure DestroyInterface; virtual;

    function GetCaps: DWord;

  public
    constructor Create( Owner: TDirectDraw; Flags: Longint;
      pColorTable: PPaletteEntry );
    destructor Destroy; override;

    procedure GetEntries( Flags: Longint; Base: Longint; NumEntries: Longint;
        pEntries: PPaletteEntry );
    procedure SetEntries( Flags: Longint; StartingEntry: Longint;
        Count: Longint; pEntries: PPaletteEntry );

    property Caps: DWord read GetCaps;
    property Interface1: IDirectDrawPalette read FInterface1;
    property Owner: TDirectDraw read FOwner;
  end;


  TDirectDrawClipper = class( TObject )
  private
    FOwner: TDirectDraw;
    FInterface1: IDirectDrawClipper;
    FWindow: HWnd;

    procedure SetWindow( Window: HWnd );

  protected
    procedure InitializeInterface( Flags: Longint ); virtual;
    procedure DestroyInterface; virtual;

    function GetIsClipListChanged: Boolean;

  public
    constructor Create( Owner: TDirectDraw; Flags: Longint );
    destructor Destroy; override;

    function GetClipList( const Rect: TRect; pClipList: PRgnData ):
        DWord;
    procedure SetClipList( pClipList: PRgnData; Flags: Longint );

    property Interface1: IDirectDrawClipper read FInterface1;
    property IsClipListChanged: Boolean read GetIsClipListChanged;
    property Owner: TDirectDraw read FOwner;
    property Window: HWnd read FWindow write SetWindow;
  end;


  TDirectDrawVideoPort = class( TObject )
  private
    FInterface1: IDirectDrawVideoPort;
    FOwner: TDirectDraw2;

    function GetColorControls: TDDColorControl;
    function GetFieldPolarity: Boolean;
    function GetVideoLine: DWord;
    function GetVideoSignalStatus: DWord;
    procedure SetColorControls( Value: TDDColorControl );

  public
    constructor Create( Owner: TDirectDraw2; Flags: Longint;
      const VideoPortDesc: TDDVideoPortDesc ); virtual;
    destructor Destroy; override;

    procedure Flip( Surface: TDirectDrawSurface; Flags: Longint );
    function GetBandwidthInfo( const PixelFormat: TDDPixelFormat; Width,
      Height, Flags: Longint ): TDDVideoPortBandwidth;
    function GetInputFormats( var NumFormats: DWord;
      pFormats: PDDPixelFormat; Flags: Longint ): Longint;
    function GetOutputFormats( const InputFormat:
      TDDPixelFormat; var NumFormats: DWord; pFormats: PDDPixelFormat;
      Flags: Longint ): Longint;
    procedure SetTargetSurface( Surface: TDirectDrawSurface;
      Flags: Longint );
    procedure StartVideo( const VideoInfo: TDDVideoPortInfo );
    procedure StopVideo;
    procedure UpdateVideo( const VideoInfo: TDDVideoPortInfo );
    function WaitForSync( Flags, Line, Timeout: Longint ): Boolean;

    property ColorControls: TDDColorControl read GetColorControls
      write SetColorControls;
    property FieldPolarity: Boolean read GetFieldPolarity;
    property Interface1: IDirectDrawVideoPort read FInterface1;
    property Owner: TDirectDraw2 read FOwner;
    property VideoLine: DWord read GetVideoLine;
    property VideoSignalStatus: DWord read GetVideoSignalStatus;
  end;


  procedure EnumerateDirectDraw( List: TList );


implementation

var
  DirectDrawList: TList;
  SurfaceList: TList;
  ClipperList: TList;
  PaletteList: TList;
  VideoPortList: TList;

  procedure DDCheck( Value: HRESULT ); forward;
  function Callback( lpGUID: PGUID ; lpDriverDescription: LPSTR ;
      lpDriverName: LPSTR ; lpContext: Pointer ): BOOL; stdcall; forward;
  function fnCallback( DDSurface: IDirectDrawSurface;
      lpContext: Pointer ): HRESULT ; stdcall; forward;
  function EnumModesCallback( const DDSurfaceDesc: TDDSurfaceDesc;
      lpContext: Pointer ): HRESULT; stdcall; forward;
  function EnumSurfacesCallback( DDSurface: IDirectDrawSurface;
      const DDSurfaceDesc: TDDSurfaceDesc; lpContext: Pointer ): HRESULT ;
      stdcall; forward;
  function EnumVideoCallback( const lpDDVideoPortCaps: TDDVideoPortCaps;
    pContext: Pointer ): HRESULT; stdcall; forward;
  function SearchClipper( DDC: IDirectDrawClipper ): TDirectDrawClipper; forward;
  function SearchPalette( DDP: IDirectDrawPalette ): TDirectDrawPalette; forward;
  function SearchSurface( DDS: IDirectDrawSurface ): TDirectDrawSurface; forward;


{$IFDEF VER90}
procedure Assert(expr : Boolean ; const msg: string);
begin
     if Expr then ShowMessage(Msg);
end;
{$ENDIF}

{**********************************************************
    TDirectDraw Object
**********************************************************}

constructor TDirectDraw.Create( lpGUID: PGUID );
begin
  inherited Create;

  // adds itself to the global list of DirectDraws
  DirectDrawList.Add( self );

  InitializeInterface( lpGUID );

  FClippers := TList.Create;
  FPalettes := TList.Create;
  FSurfaces := TList.Create;
end ;

destructor TDirectDraw.Destroy ;
begin
  // delete surfaces
  if FSurfaces <> nil then
  begin
    while FSurfaces.Count > 0 do
      TDirectDrawSurface(FSurfaces[0]).Free;
    FSurfaces.Free;
  end;

  // delete palettes
  if FPalettes <> nil then
  begin
    while FPalettes.Count > 0 do
      TDirectDrawPalette(FPalettes[0]).Free;
    FPalettes.Free;
  end;

  // delete clippers
  if FClippers <> nil then
  begin
    while FClippers.Count > 0 do
      TDirectDrawClipper(FClippers[0]).Free;
    FClippers.Free;
  end;

  if FFourCCCodes <> nil then
    FreeMem( FFourCCCodes );

  DestroyInterface;

  DirectDrawList.Remove( self );

  inherited Destroy;
end;


procedure TDirectDraw.InitializeInterface( lpGUID: PGUID );
begin
  { Initialize the DirectDraw system }
  DDCheck( DirectDrawCreate( lpGUID, FInterface1, nil ) );

  FCurInterface := FInterface1;
end;


procedure TDirectDraw.DestroyInterface;
begin
  if Assigned(FInterface1) then
    FInterface1 := nil ;
end;


procedure TDirectDraw.EnumerateDisplayModes( Flags: DWord;
  pSurfaceDesc: PDDSurfaceDesc; List: TList );
begin
  DDCheck( IDirectDraw(FCurInterface).EnumDisplayModes( Flags, pSurfaceDesc,
    Pointer(List), EnumModesCallback ) );
end;


procedure TDirectDraw.EnumerateSurfaces( Flags: Longint;
  const SurfaceDesc: TDDSurfaceDesc; List: TList );
begin
  // fetch list
  DDCheck( IDirectDraw(FCurInterface).EnumSurfaces( Flags, SurfaceDesc,
    List, EnumSurfacesCallback ) );
end;


procedure TDirectDraw.FlipToGDISurface;
begin
  DDCheck( IDirectDraw(FCurInterface).FlipToGDISurface );
end;


procedure TDirectDraw.Notification(Obj: TObject; Operation: TOperation);
begin
  if (Operation = opRemove) then
  begin
    if (Obj is TDirectDrawClipper) then
      FClippers.Remove( Obj );
    if (Obj is TDirectDrawPalette) then
      FPalettes.Remove( Obj );
    if (Obj is TDirectDrawSurface) then
      FSurfaces.Remove( Obj );
  end
  else
  begin
    if (Obj is TDirectDrawClipper) and (FClippers.IndexOf( Obj ) = -1) then
      FClippers.Add( Obj );
    if (Obj is TDirectDrawPalette) and (FPalettes.IndexOf( Obj ) = -1) then
      FPalettes.Add( Obj );
    if (Obj is TDirectDrawSurface) and (FSurfaces.IndexOf( Obj ) = -1) then
      FSurfaces.Add( Obj );
  end;
end;


procedure TDirectDraw.RestoreDisplayMode;
begin
  DDCheck( IDirectDraw(FCurInterface).RestoreDisplayMode );
end;


procedure TDirectDraw.SetCooperativeLevel( Window: HWND; Flags: Longint );
begin
  DDCheck( IDirectDraw(FCurInterface).SetCooperativeLevel( Window, Flags ) );
end;


procedure TDirectDraw.SetDisplayMode( Width: Longint; Height: Longint;
  Bpp: Longint );
begin
  DDCheck( Interface1.SetDisplayMode( Width, Height, Bpp ) );
end;


function TDirectDraw.WaitForVerticalBlank( Flags: Longint ): Boolean;
var
  r: HResult;
begin
  r := IDirectDraw(FCurInterface).WaitForVerticalBlank( Flags, 0 );
  if (r = DD_OK) then Result := True
  else Result := False;

  DDCheck( r );
end;


function TDirectDraw.GetClipper( Index: Integer ): TDirectDrawClipper;
begin
  Result := TDirectDrawClipper( FClippers[Index] );
end;


function TDirectDraw.GetClipperCount: Integer;
begin
  Result := FClippers.Count;
end;


function TDirectDraw.GetDisplayMode: TDDSurfaceDesc;
begin
  Result.dwSize := sizeof(Result);
  DDCheck( IDirectDraw(FCurInterface).GetDisplayMode( Result ) );
end;


function TDirectDraw.GetDriverCaps: TDDCaps;
var
  g: TDDCaps;
begin
  Result.dwSize := sizeof(Result);
  g.dwSize := sizeof(g);
  DDCheck( IDirectDraw(FCurInterface).GetCaps( @Result, @g ) );
end;


function TDirectDraw.GetFourCCCodes( Index: Integer ): Longint;
begin
  InitFourCCCodes;

  if (Index >= FourCCCodeCount) or (Index < 0) then
    raise EListError.Create('Index out of bounds');

  Result := PLongint(PChar(FFourCCCodes) + Index*4)^;
end;


function TDirectDraw.GetFourCCCodeCount: Integer;
begin
  InitFourCCCodes;

  Result := FFourCCCodeCount;
end;


procedure TDirectDraw.InitFourCCCodes;
begin
  // get FourCCCodes into an array
  if FFourCCCodes = nil then
  begin
    FFourCCCodeCount := 0;
    DDCheck( IDirectDraw(FCurInterface).GetFourCCCodes( FFourCCCodeCount, FFourCCCodes ) );
    GetMem( FFourCCCodes, FFourCCCodeCount*sizeof(Longint) );
    DDCheck( IDirectDraw(FCurInterface).GetFourCCCodes( FFourCCCodeCount, FFourCCCodes ) );
  end;
end;


function TDirectDraw.GetGDISurface: TDirectDrawSurface;
var
  GDIDDSurface: IDirectDrawSurface;
begin
  DDCheck( IDirectDraw(FCurInterface).GetGDISurface( GDIDDSurface ) );

  Result := SearchSurface( GDIDDSurface );
end;


function TDirectDraw.GetHELCaps: TDDCaps;
var
  g: TDDCaps;
begin
  Result.dwSize := sizeof(Result);
  g.dwSize := sizeof(g);
  DDCheck( IDirectDraw(FCurInterface).GetCaps( @g, @Result ) );
end;


function TDirectDraw.GetIsVerticalBlank: Boolean;
var
  r: BOOL;
begin
  DDCheck( IDirectDraw(FCurInterface).GetVerticalBlankStatus( r ) );
  Result := r;
end;


function TDirectDraw.GetMonitorFrequency: DWord;
begin
  DDCheck( IDirectDraw(FCurInterface).GetMonitorFrequency( Result ) );
end;


function TDirectDraw.GetPalette( Index: Integer ): TDirectDrawPalette;
begin
  Result := TDirectDrawPalette( FPalettes[Index] );
end;


function TDirectDraw.GetPaletteCount: Integer;
begin
  Result := FPalettes.Count;
end;


function TDirectDraw.GetScanLine: Longint;
var
  r: HResult;
  V: DWord;
begin
  r := IDirectDraw(FCurInterface).GetScanLine( V );
  if (r = DDERR_VERTICALBLANKINPROGRESS) then Result := -1
  else
  begin
    DDCheck( r );
    Result := V;
  end;
end;


function TDirectDraw.GetSurface( Index: Integer ): TDirectDrawSurface;
begin
  Result := TDirectDrawSurface( FSurfaces[Index] );
end;


function TDirectDraw.GetSurfaceCount: Integer;
begin
  Result := FSurfaces.Count;
end;


function TDirectDraw.LoadBitmap( const Name: String;  Width, Height: Integer ):
  TDirectDrawSurface;
//  create a DirectDrawSurface from a bitmap resource.
var
  hbm: HBITMAP;
  bm: TBITMAP;
  ddsd: TDDSurfaceDesc;
begin
    //
    //  try to load the bitmap as a resource, if that fails, try it as a file
    //
    hbm := HBITMAP(LoadImage(HInstance, Pointer(Name), IMAGE_BITMAP,
      Width, Height, LR_CREATEDIBSECTION));

    if (hbm = 0) then
      hbm := HBITMAP(LoadImage(0, Pointer(Name), IMAGE_BITMAP, Width, Height,
        LR_LOADFROMFILE + LR_CREATEDIBSECTION));

    if (hbm = 0) then
      raise EDelphiDirectDrawError.Create('Can''t find ''' + Name + ''' bitmap');

    try
      //
      // get size of the bitmap
      //
      GetObject(hbm, sizeof(bm), @bm);      // get size of bitmap

      //
      // create a DirectDrawSurface for this bitmap
      //
      ZeroMemory( @ddsd, sizeof(ddsd) );
      ddsd.dwSize := sizeof(ddsd);
      ddsd.dwFlags := DDSD_CAPS + DDSD_HEIGHT + DDSD_WIDTH;
      ddsd.DDSCaps.dwCaps := DDSCaps_OFFSCREENPLAIN;
      ddsd.dwWidth := bm.bmWidth;
      ddsd.dwHeight := bm.bmHeight;

      Result := TDirectDrawSurface.Create( self, ddsd );

      Result.CopyBitmap( hbm, 0, 0, 0, 0 );
    finally
      DeleteObject( hbm );
    end;
end;


function TDirectDraw.LoadPalette( const Name: String ): TDirectDrawPalette;
//  Create a TDirectDraw palette object from a bitmap resoure
//
//  if the resource does not exist or '' is passed create a
//  default 332 palette.
var
  i, n, fh: Integer;
  h: HRsrc;
  lpbi: PBITMAPINFOHEADER;
  ape: array[0..255] of TPALETTEENTRY;
  prgb: PRGBQUAD;
  bf: TBITMAPFILEHEADER;
  bi: TBITMAPINFOHEADER;
  r: Byte;
begin
  //
  // build a 332 palette as the default.
  //
  for i := 0 to 255 do
  begin
    ape[i].peRed    := (((i shr 5) and $07) * 255 div 7);
    ape[i].peGreen  := (((i shr 2) and $07) * 255 div 7);
    ape[i].peBlue   := (((i shr 0) and $03) * 255 div 3);
    ape[i].peFlags  := 0;
  end;

  //
  // get a pointer to the bitmap resource.
  //
  if (Name <> '') then
  begin
    h := FindResource(0, Pointer(Name), RT_BITMAP);
    if (h <> 0) then
    begin
      lpbi := PBITMAPINFOHEADER(LockResource(LoadResource(0, h)));
      if (lpbi = nil) then
        raise EDelphiDirectDrawError.Create('lock resource failed');

      prgb := PRGBQUAD(PChar(lpbi) + lpbi.biSize);

      if (lpbi = nil) or (lpbi.biSize < sizeof(TBITMAPINFOHEADER)) then
        n := 0
      else if (lpbi.biBitCount > 8) then
          n := 0
        else if (lpbi.biClrUsed = 0) then
            n := 1 shl lpbi.biBitCount
          else
              n := lpbi.biClrUsed;

      //
      //  a DIB color table has its colors stored BGR not RGB
      //  so flip them around.
      //
      for i := 0 to n-1 do
      begin
        ape[i].peRed    := PRGBQUAD(PChar(prgb) + i * sizeof(TRGBQUAD))^.rgbRed;
        ape[i].peGreen  := PRGBQUAD(PChar(prgb) + i * sizeof(TRGBQUAD))^.rgbGreen;
        ape[i].peBlue   := PRGBQUAD(PChar(prgb) + i * sizeof(TRGBQUAD))^.rgbBlue;
        ape[i].peFlags  := 0;
      end;
    end
    else
    begin
      fh := _lopen(Pointer(Name), OF_READ);
      if ( fh <> -1) then
      try
        _lread(fh, @bf, sizeof(bf));
        _lread(fh, @bi, sizeof(bi));
        _lread(fh, @ape, sizeof(ape));

        if (bi.biSize <> sizeof(TBITMAPINFOHEADER)) then
          n := 0
        else if (bi.biBitCount > 8) then
            n := 0
          else if (bi.biClrUsed = 0) then
              n := 1 shl bi.biBitCount
            else
                n := bi.biClrUsed;

        //
        //  a DIB color table has its colors stored BGR not RGB
        //  so flip them around.
        //
        for i := 0 to n-1 do
        begin
          r := ape[i].peRed;
          ape[i].peRed := ape[i].peBlue;
          ape[i].peBlue := r;
        end;
      finally
        _lclose(fh);
      end;
    end;
  end;

  Result := TDirectDrawPalette.Create(self, DDPCAPS_8BIT, @ape);
end;


//********************
// TDIRECTDRAW2 Object
//********************

constructor TDirectDraw2.Create( lpGUID: PGUID );
begin
  inherited Create( lpGUID );

  FVideoPorts := TList.Create;
end;


destructor TDirectDraw2.Destroy;
begin
  FVideoPorts.Free;

  inherited Destroy;
end;


procedure TDirectDraw2.InitializeInterface( lpGUID: PGUID );
begin
  inherited InitializeInterface( lpGUID );

  { Get the IDirectDraw2 interface }
  if Interface1.QueryInterface( IID_IDirectDraw2, FInterface2 ) <> S_OK then
    raise EDelphiDirectDrawError.Create('Unable to create IDirectDraw2 interface');

  FCurInterface := FInterface2;

  { Get the IDirect3D interface }
  if Interface2.QueryInterface( IID_IDirect3D, FDirect3D ) <> S_OK then
    raise EDelphiDirectDrawError.Create('Unable to create IDirect3D interface');

  { Get the IDDVideoPortContainer interface }
  Interface2.QueryInterface( IID_IDDVideoPortContainer, FDDVideoPortContainer );
end;


procedure TDirectDraw2.DestroyInterface;
begin
  if Assigned(FDDVideoPortContainer) then
    FDDVideoPortContainer := nil;

  if Assigned(FDirect3D) then
    FDirect3D := nil;

  if Assigned(FInterface2) then
    FInterface2 := nil ;

  inherited DestroyInterface;
end;


procedure TDirectDraw2.Notification(Obj: TObject; Operation: TOperation);
begin
  inherited Notification(Obj, Operation);

  if (Operation = opRemove) and (Obj is TDirectDrawVideoPort) then
    FVideoPorts.Remove( Obj )
  else
    if (Obj is TDirectDrawVideoPort) and (FVideoPorts.IndexOf( Obj ) = -1) then
      FVideoPorts.Add( Obj );
end;


procedure TDirectDraw2.GetAvailableVidMem ( var Caps: TDDSCaps;
  var Total, Free: DWord );
begin
  DDCheck( IDirectDraw2(FCurInterface).GetAvailableVidMem( Caps, Total, Free ) );
end;


procedure TDirectDraw2.SetDisplayMode( Width: Longint; Height: Longint;
  Bpp: Longint; RefreshRate: Longint; Flags: Longint );
begin
  DDCheck( FInterface2.SetDisplayMode( Width, Height, Bpp, RefreshRate,
    Flags ) );
end;


function TDirectDraw2.GetVideoPortStatus( PortId: Longint ): TDDVideoPortStatus;
begin
  Assert(FDDVideoPortContainer = nil, 'DDVideoPortContainer interface is nil');
  DDCheck( FDDVideoPortContainer.QueryVideoPortStatus( PortId, Result ) );
end;


function TDirectDraw2.GetVideoPort( Index: Integer ): TDirectDrawVideoPort;
begin
  Result := TDirectDrawVideoPort( FVideoPorts[Index] );
end;


function TDirectDraw2.GetVideoPortCount: Integer;
begin
  Result := FVideoPorts.Count;
end;


procedure TDirectDraw2.EnumerateVideoPorts( Flags: Longint;
  VideoPortCaps: PDDVideoPortCaps; List: TList );
begin
  Assert(FDDVideoPortContainer = nil, 'DDVideoPortContainer interface is nil');
  DDCheck( FDDVideoPortContainer.EnumVideoPorts( Flags, VideoPortCaps,
    Pointer(List), EnumVideoCallback ) );
end;


procedure TDirectDraw2.GetVideoPortConnectInfo( PortId: DWord;
  var NumEntries: DWord; pConnectInfo: PDDVideoPortConnect );
begin
  Assert(FDDVideoPortContainer = nil, 'DDVideoPortContainer interface is nil');
  DDCheck( FDDVideoPortContainer.GetVideoPortConnectInfo( PortId, NumEntries,
    pConnectInfo ) );
end;


{**********************************************************
**********************************************************
    TDirectDrawSurface Object
**********************************************************
**********************************************************}

constructor TDirectDrawSurface.Create( Owner: TDirectDraw;
  const SurfaceDesc: TDDSurfaceDesc );
var
  dc: TDDSCaps;
  i: IDirectDrawSurface;
begin
  inherited Create;

  SurfaceList.Add( self );

  FOwner := Owner;

  // create surface interface
  DDCheck( FOwner.Interface1.CreateSurface( SurfaceDesc, FInterface1, nil ) );

  InitializeInterface;

  // add itself to DirectDraw surface list
  FOwner.Notification( self, opInsert );

  // if it's a complex surface create all other surfaces
  if (SurfaceDesc.ddsCaps.dwCaps and DDSCAPS_COMPLEX) <> 0 then
  begin
     FChain := TList.Create;

    dc.dwcaps := DDSCAPS_COMPLEX;
    FInterface1.GetAttachedSurface(dc, i);
    while (i <> FInterface1) do
    begin
      FChain.Add( TDirectDrawSurface.CreateChained( Self, i ) );
      i.GetAttachedSurface(dc, i);
    end;
  end;
end;


constructor TDirectDrawSurface.CreateDuplicated( Surface: TDirectDrawSurface );
begin
  inherited Create;

  SurfaceList.Add( self );

  FOwner := Surface.Owner;

  DDCheck( Surface.Owner.Interface1.DuplicateSurface( Surface.Interface1,
    FInterface1 ) );

  InitializeInterface;

  // add itself to DirectDraw surface list
  FOwner.Notification( self, opInsert );
end;


constructor TDirectDrawSurface.CreateChained( Root: TDirectDrawSurface;
  Interf: IDirectDrawSurface );
begin
  inherited Create;

  SurfaceList.Add( self );

  FOwner := Root.Owner;

  FInterface1 := Interf;

  InitializeInterface;

  // add itself to DirectDraw surface list
  FOwner.Notification( self, opInsert );
end;


destructor TDirectDrawSurface.Destroy;
begin
  // if it's a complex surface and this is the root destroy the other surfaces
  if (FChain <> nil) then
  begin
    while FChain.Count > 0 do
    begin
      TDirectDrawSurface(FChain[0]).Free;
      FChain.Delete(0);
    end;
    FChain.Free;
  end;

  DestroyInterface;

  // remove from the DirectDraw surface list
  if (FOwner <> nil) then
    FOwner.Notification( self, opRemove );
  FOwner := nil;

  SurfaceList.Remove( self );

  inherited Destroy ;
end;


procedure TDirectDrawSurface.InitializeInterface;
begin
  FCurInterface := FInterface1;

  { Get the IDirectDrawGammaRamp interface }
  FInterface1.QueryInterface( IID_IDirectDrawGammaControl, FGammaControl );
end;


procedure TDirectDrawSurface.DestroyInterface;
begin
  if Assigned(FInterface1) then
    FInterface1 := nil;
end;


procedure TDirectDrawSurface.AddAttachedSurface( AttachedSurface:
  TDirectDrawSurface );
begin
  DDCheck( FInterface1.AddAttachedSurface(AttachedSurface.Interface1) );
end;


procedure TDirectDrawSurface.AddOverlayDirtyRect( const Rect: TRect );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).AddOverlayDirtyRect( Rect ) );
end;


procedure TDirectDrawSurface.Blt( pDestRect: PRect;
  SrcSurface: TDirectDrawSurface; pSrcRect: PRect ; Flags: Longint;
  pBltFx: PDDBLTFX );
begin
  if (SrcSurface = nil) then
    DDCheck( IDirectDrawSurface(FCurInterface).Blt( pDestRect,
      nil, pSrcRect, Flags, pBltFX ) )
  else
    DDCheck( IDirectDrawSurface(FCurInterface).Blt( pDestRect,
      SrcSurface.Interface1, pSrcRect, Flags, pBltFX ) );
end;


procedure TDirectDrawSurface.BltBatch( const BltBatch: TDDBltBatch ;
  Count: Longint; Flags: Longint );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).BltBatch( BltBatch, Count, Flags ) );
end;


procedure TDirectDrawSurface.BltFast( X: Longint; Y: Longint;
  SrcSurface: TDirectDrawSurface; pSrcRect: PRect; Trans: Longint );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).BltFast( X, Y, SrcSurface.Interface1, pSrcRect, Trans ) );
end;


procedure TDirectDrawSurface.DeleteAttachedSurface( Flags: Longint;
  AttachedSurface: TDirectDrawSurface);
begin
  if AttachedSurface = nil then
    DDCheck( IDirectDrawSurface(FCurInterface).DeleteAttachedSurface( Flags, nil ) )
  else
    DDCheck( IDirectDrawSurface(FCurInterface).DeleteAttachedSurface( Flags, AttachedSurface.Interface1 ) );
end;


procedure TDirectDrawSurface.EnumerateAttachedSurfaces ( List: TList );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).EnumAttachedSurfaces( Pointer(List), EnumSurfacesCallback ) );
end;


procedure TDirectDrawSurface.EnumerateOverlayZOrders( Flags: Longint;
  List: TList );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).EnumOverlayZOrders( Flags, Pointer(List), EnumSurfacesCallback ) );
end;


procedure TDirectDrawSurface.Flip( SurfaceTargetOverride: TDirectDrawSurface;
  Flags: Longint );
begin
  if SurfaceTargetOverride = nil then
    DDCheck( IDirectDrawSurface(FCurInterface).Flip( nil, Flags ) )
  else
    DDCheck( IDirectDrawSurface(FCurInterface).Flip( SurfaceTargetOverride.Interface1, Flags ) );
end;


function TDirectDrawSurface.GetAttachedSurface( Caps: TDDSCaps ):
  TDirectDrawSurface;
var
  dds: IDirectDrawSurface;
begin
  DDCheck( IDirectDrawSurface(FCurInterface).GetAttachedSurface( Caps, dds ) );

  Result := SearchSurface( dds );
end;


function TDirectDrawSurface.GetBltStatus( Flags: Longint ): Boolean;
var
  r: HResult;
begin
  r := IDirectDrawSurface(FCurInterface).GetBltStatus( Flags );

  if (r = DDERR_WASSTILLDRAWING) then Result := False
  else
  begin
    Result := True;
    DDCheck( r );
  end;
end;


function TDirectDrawSurface.GetColorKey( Flags: Longint ): TDDColorKey;
begin
  DDCheck( IDirectDrawSurface(FCurInterface).GetColorKey( Flags, Result ) );
end;


function TDirectDrawSurface.GetDC( var DeviceContext: HDC ): Boolean;
var
  r: HResult;
begin
  r := IDirectDrawSurface(FCurInterface).GetDC( DeviceContext );

  if (r = DDERR_WASSTILLDRAWING) then Result := False
  else
  begin
    Result := True;
    DDCheck( r );
  end;
end;


function TDirectDrawSurface.GetFlipStatus( Flags: Longint ): Boolean;
var
  r: HRESULT;
begin
  r := IDirectDrawSurface(FCurInterface).GetFlipStatus( Flags );

  if (r = DDERR_WASSTILLDRAWING) then Result := False
  else
  begin
    Result := True;
    DDCheck( r );
  end;
end;


procedure TDirectDrawSurface.Lock( pDestRect: PRect;
  var SurfaceDesc: TDDSurfaceDesc; Flags: Longint; hEvent: THandle );
begin
  SurfaceDesc.dwSize := sizeof( SurfaceDesc );
  DDCheck( IDirectDrawSurface(FCurInterface).Lock( pDestRect, SurfaceDesc, Flags, hEvent ) );
end;


procedure TDirectDrawSurface.ReleaseDC( const DeviceContext: HDC );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).ReleaseDC( DeviceContext ) );
end;


function TDirectDrawSurface.Restore: Boolean;
var
  r: HResult;
begin
  r := IDirectDrawSurface(FCurInterface).Restore;
  if (r = DDERR_WRONGMODE) then Result := False
  else
  begin
    Result := True;
    DDCheck( r );
  end;
end;


procedure TDirectDrawSurface.SetClipper( DDClipper: TDirectDrawClipper );
begin
  if DDClipper = nil then
    DDCheck( IDirectDrawSurface(FCurInterface).SetClipper( nil ) )
  else
    DDCheck( IDirectDrawSurface(FCurInterface).SetClipper( DDClipper.Interface1 ) );
end;


procedure TDirectDrawSurface.SetColorKey( Flags: Longint;
  const ColorKey: TDDColorKey );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).SetColorKey( Flags, @ColorKey ) );
end;


procedure TDirectDrawSurface.SetOverlayPosition( const Pos: TPoint );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).SetOverlayPosition( Pos.X, Pos.Y ) );
end;


procedure TDirectDrawSurface.SetPalette( DDPalette: TDirectDrawPalette );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).SetPalette( DDPalette.Interface1 ) );
end;


function TDirectDrawSurface.Unlock: Pointer;
begin
  Result := nil;
  DDCheck( IDirectDrawSurface(FCurInterface).Unlock( Result ) );
end;


procedure TDirectDrawSurface.UpdateOverlay( const SrcRect: TRect;
  DestSurface: TDirectDrawSurface; const DestRect: TRect;
  Flags: Longint; const OverlayFx: TDDOverlayFx );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).UpdateOverlay( @SrcRect, DestSurface.Interface1, @DestRect,
    Flags, @OverlayFx ) );
end;


procedure TDirectDrawSurface.UpdateOverlayDisplay( Flags: Longint );
begin
  DDCheck( IDirectDrawSurface(FCurInterface).UpdateOverlayDisplay( Flags ) );
end;


procedure TDirectDrawSurface.UpdateOverlayZOrder( Flags: Longint;
  SurfaceReference: TDirectDrawSurface );
begin
  if SurfaceReference = nil then
    DDCheck( IDirectDrawSurface(FCurInterface).UpdateOverlayZOrder( Flags, nil ) )
  else
    DDCheck( IDirectDrawSurface(FCurInterface).UpdateOverlayZOrder( Flags, SurfaceReference.Interface1 ) );
end;


function TDirectDrawSurface.GetCaps: TDDSCaps;
begin
  DDCheck( IDirectDrawSurface(FCurInterface).GetCaps( Result ) );
end;

{procedure TDirectDrawSurface.GetGammaRamp( Flags: DWord;
  var RampData: TGammaRamp );
begin
  DDCheck( FGammaControl.GetGammaRamp( Flags, RampData ) );
end;
}
function TDirectDrawSurface.GetClipper: TDirectDrawClipper;
var
  ddc: IDirectDrawClipper;
  r: HResult;
begin
  // get clipper interface
  r := IDirectDrawSurface(FCurInterface).GetClipper( ddc );
  if (r = DDERR_NOCLIPPERATTACHED) then
    Result := nil
  else
  begin
    DDCheck( r );

    Result := SearchClipper( ddc );
  end;
end;


function TDirectDrawSurface.GetOverlayPosition: TPoint;
begin
  DDCheck( IDirectDrawSurface(FCurInterface).GetOverlayPosition( Result.X, Result.Y ) );
end;


function TDirectDrawSurface.GetPalette: TDirectDrawPalette;
var
  ddp: IDirectDrawPalette;
begin
  DDCheck( IDirectDrawSurface(FCurInterface).GetPalette( ddp ) );

  Result := SearchPalette( ddp );
end;


function TDirectDrawSurface.GetPixelFormat: TDDPixelFormat;
begin
  Result.dwSize := sizeof( Result );
  DDCheck( IDirectDrawSurface(FCurInterface).GetPixelFormat( Result ) );
end;


function TDirectDrawSurface.GetSurfaceDesc: TDDSurfaceDesc;
begin
  Result.dwSize := sizeof( Result );
  DDCheck( IDirectDrawSurface(FCurInterface).GetSurfaceDesc( Result ) );
end;


function TDirectDrawSurface.GetIsLost: Boolean;
var
  r: HRESULT;
begin
  r := IDirectDrawSurface(FCurInterface).IsLost;

  if (r = DDERR_SURFACELOST) then Result := True
  else
  begin
    Result := False;
    DDCheck( r );
  end;
end;


function TDirectDrawSurface.ColorMatchRGB( rgb: COLORREF ): Longint;
var
  rgbT: COLORREF;
  dc: HDC;
  dw: DWord;
  ddsd: TDDSurfaceDesc;
begin
  dw := CLR_INVALID;
  rgbT := CLR_INVALID;

  //
  //  use GDI SetPixel to color match for us
  //
  if (rgb <> CLR_INVALID) then
  begin
    GetDC( dc );
    try
      rgbT := GetPixel(dc, 0, 0);             // save current pixel value
      SetPixel(dc, 0, 0, rgb);               // set our value
    finally
      ReleaseDC(dc);
    end;
  end;

  //
  // now lock the surface so we can read back the converted color
  //
  ddsd.dwSize := sizeof(ddsd);
  try
    while True do
      try
        Lock(nil, ddsd, 0, 0);
        break;
      except
        on E: EDirectDrawError do
          if E.ErrorCode <> DDERR_WASSTILLDRAWING then raise;
      end;

    dw := PLongint(ddsd.lpSurface)^;                     // get DWORD
    dw := dw and ((1 shl ddsd.ddpfPixelFormat.dwRGBBitCount)-1);  // mask it to bpp
    Unlock;
  except
    on EDirectDrawError do begin end;
  end;

  //
  //  now put the color that was there back.
  //
  if (rgb <> CLR_INVALID) then
  begin
    GetDC( dc );
    SetPixel(dc, 0, 0, rgbT);
    ReleaseDC(dc);
  end;

  Result := dw
end;


procedure TDirectDrawSurface.CopyBitmap( BitmapHandle: HBITMAP; Left, Top,
  Width, Height: Integer );
var
  dcImage, dc: HDC;
  bm: TBITMAP;
  ddsd: TDDSurfaceDesc;
begin
  if (BitmapHandle = 0) then
    raise EDelphiDirectDrawError.Create('Bitmap handle is nil');

  //
  // make sure this surface is restored.
  //
  Restore;

  //
  //  select bitmap into a memoryDC so we can use it.
  //
  dcImage := CreateCompatibleDC(0);
  if (dcImage = 0) then
    raise EDelphiDirectDrawError.Create('Failed to CreateCompatible DC');
  try
    SelectObject(dcImage, BitmapHandle);

    //
    // get size of the bitmap
    //
    GetObject(BitmapHandle, sizeof(bm), @bm);    // get size of bitmap
    if (Width = 0) then Width := bm.bmWidth;    // use the passed size, unless zero
    if (Height = 0) then Height := bm.bmHeight;

    //
    // get size of surface.
    //
    ddsd.dwSize := sizeof(ddsd);
    ddsd.dwFlags := DDSD_HEIGHT + DDSD_WIDTH;
    ddsd := SurfaceDesc;

    GetDC( dc );
    try
      StretchBlt( dc, 0, 0, ddsd.dwWidth, ddsd.dwHeight, dcImage, Left, Top,
        Width, Height, SRCCOPY );
    finally
      ReleaseDC( dc );
    end;
  finally
    DeleteDC( dcImage );
  end;
end;


procedure TDirectDrawSurface.ReLoadBitmap( const Name: String );
var
  hbm: HBITMAP;
begin
//
  //  try to load the bitmap as a resource, if that fails, try it as a file
  //
  hbm := HBITMAP(LoadImage(HInstance, Pointer(Name), IMAGE_BITMAP,
    0, 0, LR_CREATEDIBSECTION));

  if (hbm = 0) then
    hbm := HBITMAP( LoadImage(0, Pointer(Name), IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE + LR_CREATEDIBSECTION) );

  if (hbm = 0) then
    raise EDelphiDirectDrawError.Create('Can''t find ''' + Name + ''' bitmap');

  try
    CopyBitmap( hbm, 0, 0, 0, 0 );
  finally
    DeleteObject(hbm);
  end;
end;

{procedure TDirectDrawSurface.SetGammaRamp( Flags: DWord;
  const RampData: TGammaRamp );
begin
  DDCheck( FGammaControl.SetGammaRamp( Flags, RampData ) );
end;
}
procedure TDirectDrawSurface.SetRGBColorKey( rgb: COLORREF );
var
  ddck: TDDColorKey;
begin
    ddck.dwColorSpaceLowValue := ColorMatchRGB( rgb );
    ddck.dwColorSpaceHighValue := ddck.dwColorSpaceLowValue;

    SetColorKey(DDCKEY_SRCBLT, ddck);
end;


procedure TDirectDrawSurface2.InitializeInterface;
begin
  { Get the IDirectDrawSurface2 interface }
  if Interface1.QueryInterface( IID_IDirectDrawSurface2, FInterface2 ) <> S_OK then
    raise EDelphiDirectDrawError.Create('Unable to create IDirectDrawSurface2 interface');

  FCurInterface := FInterface2;

  { Get the IDirect3DTexture interface }
  if Interface2.QueryInterface( IID_IDirect3DTexture, FDirect3DTexture ) <> S_OK then
    raise EDelphiDirectDrawError.Create('Unable to create IDirect3DTexture interface');
end;


procedure TDirectDrawSurface2.DestroyInterface;
begin
  if Assigned(FDirect3DTexture) then
    FDirect3DTexture := nil;

  if Assigned(FInterface2) then
    FInterface2 := nil;

  inherited DestroyInterface;
end;


procedure TDirectDrawSurface2.PageLock( Flags: Longint );
begin
  DDCheck( IDirectDrawSurface2(FCurInterface).PageLock( Flags ) );
end;


procedure TDirectDrawSurface2.PageUnlock( Flags: Longint );
begin
  DDCheck( IDirectDrawSurface2(FCurInterface).PageUnlock( Flags ) );
end;

procedure TDirectDrawSurface2.AddAttachedSurface( AttachedSurface:
  TDirectDrawSurface2 );
begin
  DDCheck( IDirectDrawSurface2(FCurInterface).AddAttachedSurface(AttachedSurface.Interface2) );
end;


procedure TDirectDrawSurface3.InitializeInterface;
begin
  { Get the IDirectDrawSurface3 interface }
  if Interface1.QueryInterface( IID_IDirectDrawSurface3, FInterface3 ) <> S_OK then
    raise EDelphiDirectDrawError.Create('Unable to create IDirectDrawSurface3 interface');

  FCurInterface := FInterface3;
end;


procedure TDirectDrawSurface3.DestroyInterface;
begin
  if Assigned(FInterface3) then
    FInterface3 := nil;

  inherited DestroyInterface;
end;


procedure TDirectDrawSurface3.AddAttachedSurface( AttachedSurface:
  TDirectDrawSurface3 );
begin
  DDCheck( IDirectDrawSurface3(FCurInterface).AddAttachedSurface(AttachedSurface.Interface3) );
end;


procedure TDirectDrawSurface3.SetSurfaceDesc( const SurfaceDesc:
  TDDSurfaceDesc );
begin
  DDCheck( IDirectDrawSurface3(FCurInterface).SetSurfaceDesc( SurfaceDesc, 0 ) );
end;


function TDirectDrawSurface3.GetSurfaceDesc: TDDSurfaceDesc;
begin
  Result := inherited SurfaceDesc;
end;


{**********************************************************
**********************************************************
    TDirectDrawPalette Object
**********************************************************
**********************************************************}

constructor TDirectDrawPalette.Create( Owner: TDirectDraw; Flags: Longint;
  pColorTable: PPaletteEntry );
begin
  inherited Create;

  PaletteList.Add( self );

  FOwner := Owner;

  InitializeInterface( Flags, pColorTable );

  // add itself to DirectDraw palette list
  FOwner.Notification( self, opInsert );
end;


destructor TDirectDrawPalette.Destroy;
begin
  // remove from the DirectDraw palette list
  FOwner.Notification( self, opRemove );
  FOwner := nil;

  PaletteList.Remove( self );

  DestroyInterface;

  inherited Destroy ;
end;


procedure TDirectDrawPalette.InitializeInterface( Flags: Longint;
  lpColorTable: PPaletteEntry );
begin
  // create surface interface
  DDCheck( FOwner.Interface1.CreatePalette( Flags, lpColorTable,
    FInterface1, nil ) );
end;


procedure TDirectDrawPalette.DestroyInterface;
begin
  if Assigned(FInterface1) then
    FInterface1 := nil;
end;


procedure TDirectDrawPalette.GetEntries( Flags: Longint; Base: Longint;
  NumEntries: Longint; pEntries: PPaletteEntry );
begin
  DDCheck( FInterface1.GetEntries( Flags, Base, NumEntries, pEntries ) );
end;


procedure TDirectDrawPalette.SetEntries( Flags: Longint;
  StartingEntry: Longint; Count: Longint; pEntries: PPaletteEntry );
begin
  DDCheck( FInterface1.SetEntries( Flags, StartingEntry, Count, pEntries ) );
end;


function TDirectDrawPalette.GetCaps: DWord;
begin
  DDCheck( FInterface1.GetCaps( Result ) );
end;


{**********************************************************
**********************************************************
    TDirectDrawClipper Object
**********************************************************
**********************************************************}

constructor TDirectDrawClipper.Create( Owner: TDirectDraw; Flags: Longint );
begin
  inherited Create;

  ClipperList.Add( self );

  FOwner := Owner;

  InitializeInterface( Flags );

  if FOwner <> nil then
    // add itself to DirectDraw clipper list
    FOwner.Notification( self, opInsert );
end;


destructor TDirectDrawClipper.Destroy;
begin
  // remove from the DirectDraw clipper list
  if (FOwner <> nil) then
  begin
    FOwner.Notification( self, opRemove );
    FOwner := nil;
  end;

  ClipperList.Remove( self );

  DestroyInterface;

  inherited Destroy ;
end;


procedure TDirectDrawClipper.InitializeInterface( Flags: Longint );
begin
  // create clipper interface
  if FOwner = nil then
  begin
    Assert(@DirectDrawCreateClipper = nil,
      'DirectDrawCreateClipper function not available in DLL.'
      + #13 + 'You need a higher version.');
    DDCheck( DirectDrawCreateClipper( Flags, FInterface1, nil ) );
  end
  else
    DDCheck( FOwner.Interface1.CreateClipper( Flags, FInterface1, nil ) );
end;


procedure TDirectDrawClipper.DestroyInterface;
begin
  if Assigned(FInterface1) then
    FInterface1 := nil;
end;


function TDirectDrawClipper.GetClipList( const Rect: TRect;
  pClipList: PRgnData ): DWord;
begin
  DDCheck( FInterface1.GetClipList( @Rect, pClipList, Result ) );
end;


procedure TDirectDrawClipper.SetClipList( pClipList: PRgnData; Flags: Longint );
begin
  DDCheck( FInterface1.SetClipList( pClipList, Flags ) );
end;


procedure TDirectDrawClipper.SetWindow( Window: HWnd );
begin
  DDCheck( FInterface1.SetHWnd( 0, Window ) )
end;


function TDirectDrawClipper.GetIsClipListChanged;
var
  r: BOOL;
begin
  DDCheck( FInterface1.IsClipListChanged( r ) );
  Result := r;
end;



{**********************************************************
**********************************************************
    TDirectDrawVideoPort Object
**********************************************************
**********************************************************}

constructor TDirectDrawVideoPort.Create( Owner: TDirectDraw2; Flags: Longint;
  const VideoPortDesc: TDDVideoPortDesc );
begin
  inherited Create;

  // create videoport interface
  FOwner.DDVideoPortContainer.CreateVideoPort( Flags, VideoPortDesc,
    FInterface1, nil );

  VideoPortList.Add( self );

  FOwner := Owner;

  FOwner.Notification( self, opInsert );
end;


destructor TDirectDrawVideoPort.Destroy;
begin
  // remove from the DirectDraw videoport list
  if (FOwner <> nil) then
  begin
    FOwner.Notification( self, opRemove );
    FOwner := nil;
  end;

  if Assigned(FInterface1) then
    FInterface1 := nil;

  VideoPortList.Remove( self );

  inherited Destroy ;
end;


procedure TDirectDrawVideoPort.Flip( Surface: TDirectDrawSurface;
  Flags: Longint );
begin
  DDCheck( FInterface1.Flip( Surface.Interface1, Flags ) );
end;


function TDirectDrawVideoPort.GetBandwidthInfo( const PixelFormat:
  TDDPixelFormat; Width, Height, Flags: Longint ): TDDVideoPortBandwidth;
begin
  DDCheck( FInterface1.GetBandwidthInfo( PixelFormat, Width, Height, Flags,
    Result ) );
end;


function TDirectDrawVideoPort.GetInputFormats( var NumFormats: DWord;
  pFormats: PDDPixelFormat; Flags: Longint ): Longint;
var
  r: HResult;
begin
  r := FInterface1.GetInputFormats( NumFormats, pFormats, Flags );
  if (r <> DD_OK) and (r <> DDERR_MOREDATA) then
    DDCheck( r );
  Result := NumFormats;
end;


function TDirectDrawVideoPort.GetOutputFormats( const InputFormat:
  TDDPixelFormat; var NumFormats: DWord; pFormats: PDDPixelFormat;
  Flags: Longint ): Longint;
var
  r: HResult;
begin
  r := FInterface1.GetOutputFormats( InputFormat, NumFormats, pFormats, Flags );
  if (r <> DD_OK) and (r <> DDERR_MOREDATA) then
    DDCheck( r );
  Result := NumFormats;
end;


procedure TDirectDrawVideoPort.SetTargetSurface( Surface: TDirectDrawSurface;
  Flags: Longint );
begin
  DDCheck( FInterface1.SetTargetSurface( Surface.Interface1, Flags ) );
end;


procedure TDirectDrawVideoPort.StartVideo( const VideoInfo: TDDVideoPortInfo );
begin
  DDCheck( FInterface1.StartVideo( VideoInfo ) );
end;


procedure TDirectDrawVideoPort.StopVideo;
begin
  DDCheck( FInterface1.StopVideo );
end;


procedure TDirectDrawVideoPort.UpdateVideo( const VideoInfo: TDDVideoPortInfo );
begin
  DDCheck( FInterface1.UpdateVideo( VideoInfo ) );
end;


function TDirectDrawVideoPort.WaitForSync( Flags, Line, Timeout: Longint ):
  Boolean;
var
  r: HResult;
begin
  r := FInterface1.WaitForSync( Flags, Line, Timeout );
  if (r = DDERR_WASSTILLDRAWING) then Result := False
  else
  begin
    Result := True;
    DDCheck( r );
  end;
end;


function TDirectDrawVideoPort.GetColorControls: TDDColorControl;
begin
  DDCheck( FInterface1.GetColorControls( Result ) );
end;


procedure TDirectDrawVideoPort.SetColorControls( Value: TDDColorControl );
begin
  DDCheck( FInterface1.SetColorControls( Value ) );
end;


function TDirectDrawVideoPort.GetFieldPolarity: Boolean;
var
  r: BOOL;
begin
  DDCheck( FInterface1.GetFieldPolarity( r ) );
  Result := r;
end;


function TDirectDrawVideoPort.GetVideoLine: DWord;
begin
  DDCheck( FInterface1.GetVideoLine( Result ) );
end;


function TDirectDrawVideoPort.GetVideoSignalStatus: DWord;
begin
  DDCheck( FInterface1.GetVideoSignalStatus( Result ) );
end;


{**********************************************************
**********************************************************
        miscellanious functions
**********************************************************
**********************************************************}

procedure EnumerateDirectDraw( List: TList );
begin
  DDCheck( DirectDrawEnumerate( Callback, Pointer(List) ) );
end;


function EnumModesCallback( const DDSurfaceDesc: TDDSurfaceDesc;
  lpContext: Pointer ): HRESULT; stdcall;
var
  p: PDisplayModeEnum;
begin
  // add display mode to the list
  New( p );
  p.SurfaceDesc := DDSurfaceDesc;
  TList(lpContext).Add( p );

  // Next please
  Result := DDENUMRET_OK;
end;


function EnumCallback( DDSurface: IDirectDrawSurface ;
  const TDDSurfaceDesc: TDDSurfaceDesc ; lpContext: Pointer ): HRESULT ;
  stdcall ;
begin
  // add surface to the list
  TList(lpContext).Add( SearchSurface( DDSurface ) );

  // Next please
  Result := DDENUMRET_OK;
end;


function EnumSurfacesCallback( DDSurface: IDirectDrawSurface;
  const DDSurfaceDesc: TDDSurfaceDesc; lpContext: Pointer ): HRESULT; stdcall;
begin
  // add surface to the list
  TList(lpContext).Add( SearchSurface( DDSurface ) );

  // Next please
  Result := DDENUMRET_OK;
end;


function fnCallback( DDSurface: IDirectDrawSurface; lpContext: Pointer ):
  HRESULT ; stdcall ;
begin
  // add surface to the list
  TList(lpContext).Add( Pointer(DDSurface) );

  // Next please
  Result := DDENUMRET_OK;
end;


function Callback( lpGUID: PGUID ; lpDriverDescription: LPSTR ;
  lpDriverName: LPSTR ; lpContext: Pointer ): BOOL; stdcall ;
var
  p: PDirectDrawEnum;
begin
  // add surface to the list
  New( p );
  if lpGUID = nil then
    ZeroMemory( @(p.GUID), sizeof(p.GUID) )
  else
    p.GUID := lpGUID^;
  p.DriverDescription := lpDriverDescription;
  p.DriverName := lpDriverName;
  TList(lpContext).Add( p );

  // Next please
  Result := BOOL(DDENUMRET_OK);
end;


function EnumVideoCallback( const lpDDVideoPortCaps: TDDVideoPortCaps;
  pContext: Pointer ): HRESULT; stdcall ;
begin
  // add videoport to the list
  TList(pContext).Add( @lpDDVideoPortCaps );

  // Next please
  Result := DDENUMRET_OK;
end;


function SearchSurface( DDS: IDirectDrawSurface ): TDirectDrawSurface;
var
  Cnt: Integer;
begin
  for Cnt := SurfaceList.Count-1 downto 0 do
    if TDirectDrawSurface(SurfaceList[Cnt]).Interface1 = DDS then
    begin
      Result := SurfaceList[Cnt];
      exit;
    end;

  raise EDelphiDirectDrawError.Create(
    'Couldn''t find TDirectDrawSurface object!' + #13 +
    'If you created all surfaces through the TDirectDrawSurface object' + #13 +
    'this must be a bug in DelphiDirectDraw.' + #13 +
    'Please report it (preferably with sample code) to: CarlosBarbosa@bigfoot.com');
end;


function SearchClipper( DDC: IDirectDrawClipper ): TDirectDrawClipper;
var
  Cnt: Integer;
begin
  for Cnt := ClipperList.Count-1 downto 0 do
    if TDirectDrawClipper(ClipperList[Cnt]).Interface1 = DDC then
    begin
      Result := ClipperList[Cnt];
      exit;
    end;

  raise EDelphiDirectDrawError.Create(
    'Couldn''t find TDirectDrawClipper object!' + #13 +
    'If you created all clippers through the TDirectDrawClipper object' + #13 +
    'this must be a bug in DelphiDirectDraw.' + #13 +
    'Please report it (preferably with sample code) to: CarlosBarbosa@bigfoot.com');
end;


function SearchPalette( DDP: IDirectDrawPalette ): TDirectDrawPalette;
var
  Cnt: Integer;
begin
  for Cnt := PaletteList.Count-1 downto 0 do
    if TDirectDrawPalette(PaletteList[Cnt]).Interface1 = DDP then
    begin
      Result := PaletteList[Cnt];
      exit;
    end;

  raise EDelphiDirectDrawError.Create(
    'Couldn''t find TDirectDrawPalette object!' + #13 +
    'If you created all palettes through the TDirectDrawPalette object' + #13 +
    'this must be a bug in DelphiDirectDraw.' + #13 +
    'Please report it (preferably with sample code) to: CarlosBarbosa@bigfoot.com');
end;


procedure DDCheck ( Value: HRESULT ); { Check the result of a COM operation }
var
  S: String ;
begin
  if Value <> DD_OK then
  begin
    Case Value of
        DDERR_ALREADYINITIALIZED: S:='This object is already initialized.';
        DDERR_BLTFASTCANTCLIP: S:=' if a clipper object is attached to the source surface passed into a BltFast call.';
        DDERR_CANNOTATTACHSURFACE: S:='This surface can not be attached to the requested surface.';
        DDERR_CANNOTDETACHSURFACE: S:='This surface can not be detached from the requested surface.';
        DDERR_CANTCREATEDC: S:='Windows can not create any more DCs.';
        DDERR_CANTDUPLICATE: S:='Cannot duplicate primary & 3D surfaces, or surfaces that are implicitly created.';
        DDERR_CANTLOCKSURFACE: S:=' Access to this surface is being refused because no driver exists which can supply a pointer to the surface.';
        DDERR_CANTPAGELOCK: S:='The attempt to page lock a surface failed.';
        DDERR_CANTPAGEUNLOCK: S:='The attempt to page unlock a surface failed.';
        DDERR_CLIPPERISUSINGHWND: S:='An attempt was made to set a cliplist for a clipper object that is already monitoring an hwnd.';
        DDERR_COLORKEYNOTSET: S:='No src color key specified for this operation.';
        DDERR_CURRENTLYNOTAVAIL: S:='Support is currently not available.';
        DDERR_DCALREADYCREATED: S:='A DC has already been returned for this surface. Only one DC can be retrieved per surface.';
        DDERR_DIRECTDRAWALREADYCREATED: S:='A DirectDraw object representing this driver has already been created for this process.';
        DDERR_DEVICEDOESNTOWNSURFACE: S:='Surfaces created by one direct draw device cannot be used directly by another direct draw device.';
        DDERR_EXCEPTION: S:='An exception was encountered while performing the requested operation.';
        DDERR_EXCLUSIVEMODEALREADYSET: S:='An attempt was made to set the cooperative level when it was already set to exclusive.';
        DDERR_GENERIC: S:='Generic failure.';
        DDERR_HEIGHTALIGN: S:='Height of rectangle provided is not a multiple of reqd alignment.';
        DDERR_HWNDALREADYSET: S:='The CooperativeLevel HWND has already been set. It can not be reset while the process has surfaces or palettes created.';
        DDERR_HWNDSUBCLASSED: S:='HWND used by DirectDraw CooperativeLevel has been subclassed, this prevents DirectDraw from restoring state.';
        DDERR_IMPLICITLYCREATED: S:='This surface can not be restored because it is an implicitly created surface.';
        DDERR_INCOMPATIBLEPRIMARY: S:='Unable to match primary surface creation request with existing primary surface.';
        DDERR_INVALIDCAPS: S:='One or more of the caps bits passed to the callback are incorrect.';
        DDERR_INVALIDCLIPLIST: S:='DirectDraw does not support the provided cliplist.';
        DDERR_INVALIDDIRECTDRAWGUID: S:='The GUID passed to DirectDrawCreate is not a valid DirectDraw driver identifier.';
        DDERR_INVALIDMODE: S:='DirectDraw does not support the requested mode.';
        DDERR_INVALIDOBJECT: S:='DirectDraw received a pointer that was an invalid DIRECTDRAW object.';
        DDERR_INVALIDPARAMS: S:='One or more of the parameters passed to the function are incorrect.';
        DDERR_INVALIDPIXELFORMAT: S:='The pixel format was invalid as specified.';
        DDERR_INVALIDPOSITION: S:='Returned when the position of the overlay on the destination is no longer legal for that destination.';
        DDERR_INVALIDRECT: S:='Rectangle provided was invalid.';
        DDERR_INVALIDSURFACETYPE: S:='The requested action could not be performed because the surface was of the wrong type.';
        DDERR_LOCKEDSURFACES: S:='Operation could not be carried out because one or more surfaces are locked.';
        DDERR_MOREDATA: S:='There is more data available than the specified buffer size can hold.';
        DDERR_NO3D: S:='There is no 3D present.';
        DDERR_NOALPHAHW: S:='Operation could not be carried out because there is no alpha accleration hardware present or available.';
        DDERR_NOBLTHW: S:='No blitter hardware present.';
        DDERR_NOCLIPLIST: S:='No cliplist available.';
        DDERR_NOCLIPPERATTACHED: S:='No clipper object attached to surface object.';
        DDERR_NOCOLORCONVHW: S:='Operation could not be carried out because there is no color conversion hardware present or available.';
        DDERR_NOCOLORKEY: S:='Surface does not currently have a color key';
        DDERR_NOCOLORKEYHW: S:='Operation could not be carried out because there is no hardware support of the destination color key.';
        DDERR_NOCOOPERATIVELEVELSET: S:='Create function called without DirectDraw object method SetCooperativeLevel being called.';
        DDERR_NODC: S:='No DC was ever created for this surface.';
        DDERR_NODDROPSHW: S:='No DirectDraw ROP hardware.';
        DDERR_NODIRECTDRAWHW: S:='A hardware-only DirectDraw object creation was attempted but the driver did not support any hardware.';
        DDERR_NODIRECTDRAWSUPPORT: S:='DirectDraw support is not possible with the current display driver.';
        DDERR_NOEMULATION: S:='Software emulation not available.';
        DDERR_NOEXCLUSIVEMODE: S:='Operation requires the application to have exclusive mode but the application does not have exclusive mode.';
        DDERR_NOFLIPHW: S:='Flipping visible surfaces is not supported.';
        DDERR_NOGDI: S:='There is no GDI present.';
        DDERR_NOHWND: S:='Clipper notification requires an HWND or no HWND has previously been set as the CooperativeLevel HWND.';
        DDERR_NOMIPMAPHW: S:='Operation could not be carried out because there is no mip-map texture mapping hardware present or available.';
        DDERR_NOMIRRORHW: S:='Operation could not be carried out because there is no hardware present or available.';
        DDERR_NONONLOCALVIDMEM: S:='An attempt was made to allocate non-local video memory from a device that does not support non-local video memory.';
        DDERR_NOOPTIMIZEHW: S:='The device does not support optimized surfaces.';
        DDERR_NOOVERLAYDEST: S:='Returned when GetOverlayPosition is called on an overlay that UpdateOverlay has never been called on to establish a destination.';
        DDERR_NOOVERLAYHW: S:='Operation could not be carried out because there is no overlay hardware present or available.';
        DDERR_NOPALETTEATTACHED: S:='No palette object attached to this surface.';
        DDERR_NOPALETTEHW: S:='No hardware support for 16 or 256 color palettes.';
        DDERR_NORASTEROPHW: S:='Operation could not be carried out because there is no appropriate raster op hardware present or available.';
        DDERR_NOROTATIONHW: S:='Operation could not be carried out because there is no rotation hardware present or available.';
        DDERR_NOSTRETCHHW: S:='Operation could not be carried out because there is no hardware support for stretching.';
        DDERR_NOT4BITCOLOR: S:='DirectDrawSurface is not in 4 bit color palette and the requested operation requires 4 bit color palette.';
        DDERR_NOT4BITCOLORINDEX: S:='DirectDrawSurface is not in 4 bit color index palette and the requested operation requires 4 bit color index palette.';
        DDERR_NOT8BITCOLOR: S:='DirectDrawSurface is not in 8 bit color mode and the requested operation requires 8 bit color.';
        DDERR_NOTAOVERLAYSURFACE: S:='Returned when an overlay member is called for a non-overlay surface.';
        DDERR_NOTEXTUREHW: S:='Operation could not be carried out because there is no texture mapping hardware present or available.';
        DDERR_NOTFLIPPABLE: S:='An attempt has been made to flip a surface that is not flippable.';
        DDERR_NOTFOUND: S:='Requested item was not found.';
        DDERR_NOTINITIALIZED: S:='An attempt was made to invoke an interface member of a DirectDraw object created by CoCreateInstance() before it was initialized.';
        DDERR_NOTLOADED: S:='The surface is an optimized surface, but it has not yet been allocated any memory.';
        DDERR_NOTLOCKED: S:='Surface was not locked.  An attempt to unlock a surface that was not locked at all, or by this process, has been attempted.';
        DDERR_NOTPAGELOCKED: S:='An attempt was made to page unlock a surface with no outstanding page locks.';
        DDERR_NOTPALETTIZED: S:='The surface being used is not a palette-based surface.';
        DDERR_NOVSYNCHW: S:='Operation could not be carried out because there is no hardware support for vertical blank synchronized operations.';
        DDERR_NOZBUFFERHW: S:='Operation could not be carried out because there is no hardware support for zbuffer blitting.';
        DDERR_NOZOVERLAYHW: S:='Overlay surfaces could not be z layered based on their BltOrder because the hardware does not support z layering of overlays.';
        DDERR_OUTOFCAPS: S:='The hardware needed for the requested operation has already been allocated.';
        DDERR_OUTOFMEMORY: S:='DirectDraw does not have enough memory to perform the operation.';
        DDERR_OUTOFVIDEOMEMORY: S:='DirectDraw does not have enough memory to perform the operation.';
        DDERR_OVERLAYCANTCLIP: S:='The hardware does not support clipped overlays.';
        DDERR_OVERLAYCOLORKEYONLYONEACTIVE: S:='Can only have ony color key active at one time for overlays.';
        DDERR_OVERLAYNOTVISIBLE: S:='Returned when GetOverlayPosition is called on a hidden overlay.';
        DDERR_PALETTEBUSY: S:='Access to this palette is being refused because the palette is already locked by another thread.';
        DDERR_PRIMARYSURFACEALREADYEXISTS: S:='This process already has created a primary surface.';
        DDERR_REGIONTOOSMALL: S:='Region passed to Clipper::GetClipList is too small.';
        DDERR_SURFACEALREADYATTACHED: S:='This surface is already attached to the surface it is being attached to.';
        DDERR_SURFACEALREADYDEPENDENT: S:='This surface is already a dependency of the surface it is being made a dependency of.';
        DDERR_SURFACEBUSY: S:='Access to this surface is being refused because the surface is already locked by another thread.';
        DDERR_SURFACEISOBSCURED: S:='Access to surface refused because the surface is obscured.';
        DDERR_SURFACELOST: S:='Access to this surface is being refused because the surface memory is gone. The DirectDrawSurface object representing this surface should have Restore called on it.';
        DDERR_SURFACENOTATTACHED: S:='The requested surface is not attached.';
        DDERR_TOOBIGHEIGHT: S:='Height requested by DirectDraw is too large.';
        DDERR_TOOBIGSIZE: S:='Size requested by DirectDraw is too large, but the individual height and width are OK.';
        DDERR_TOOBIGWIDTH: S:='Width requested by DirectDraw is too large.';
        DDERR_UNSUPPORTED: S:='Action not supported.';
        DDERR_UNSUPPORTEDFORMAT: S:='FOURCC format requested is unsupported by DirectDraw.';
        DDERR_UNSUPPORTEDMASK: S:='Bitmask in the pixel format requested is unsupported by DirectDraw.';
        DDERR_UNSUPPORTEDMODE: S:='The display is currently in an unsupported mode.';
        DDERR_VERTICALBLANKINPROGRESS: S:='Vertical blank is in progress.';
        DDERR_VIDEONOTACTIVE: S:='The video port is not active.';
        DDERR_WASSTILLDRAWING: S:='Informs DirectDraw that the previous Blt which is transfering information to or from this Surface is incomplete.';
        DDERR_WRONGMODE: S:='This surface can not be restored because it was created in a different mode.';
        DDERR_XALIGN: S:='Rectangle provided was not horizontally aligned on required boundary.';
        Else S:='Unrecognized error value.';
    end;

    S:= Format ( 'DirectDraw call failed: %x', [ Value ] )  + #13 + S;
    raise EDirectDrawError.Create( S, Value );
  end;
end ;


initialization
begin
  DirectDrawList := TList.Create;
  SurfaceList := TList.Create;
  PaletteList := TList.Create;
  ClipperList := TList.Create;
  VideoPortList := TList.Create;
end;


finalization
begin
  VideoPortList.Free;
  ClipperList.Free;
  PaletteList.Free;
  SurfaceList.Free;
  DirectDrawList.Free;
end;

end.

