unit LensFlare;

interface

uses
  CgTypes, CgGeometry, GL;

type
  TLFFlareElement = (feGlow, feRing, feStreaks, feRays, feSecondaries);
  { The actual gradients between two colors are, of course, calculated by OpenGL.
    The start and end colors of a gradient are stored to represent the color of
    lens flare elements. }
  TLFGradient = record
    CFrom, CTo: TCGColorF;
  end;
  TLFLensFlare = class(TObject)
  public
    Size: Single;              // Radius of the flare.
    Squeeze: Single;           // To create elliptic flares.
    Seed: Integer;             // Random seed.
    NumStreaks: Integer;       // Number of streaks.
    StreakWidth: Single;       // Width of the streaks.
    NumSecs: Integer;          // Number of secondary flares.
    Resolution: Integer;       // Number of segments used when rendering circles.
    Position: TCGVector;       // Main flare's origin.
    Elements: set of TLFFlareElement;  // Which elements should be rendered?
    Gradients: array [TLFFlareElement] of TLFGradient;  // And what color should they have? 
    constructor Create;
    procedure Render;
  end;

implementation

function lfGradient(c1, c2: TCGColorF): TLFGradient;
begin

  with Result do
  begin
    CFrom := c1;
    CTo := c2;
  end;

end;

{ TLensFlare }

constructor TLFLensFlare.Create;
begin

  inherited Create;
  // Set default parameters:
  Size := 1;
  Squeeze := 1;
  Seed := 0;
  NumStreaks := 4;
  StreakWidth := 2;
  NumSecs := 8;
  Position := cgOrigin;
  Resolution := 64;
  // Render all elements by default.
  Elements := [feGlow, feRing, feStreaks, feRays, feSecondaries];
  // Setup default gradients:
  Gradients[feGlow] := lfGradient(cgColorF(1, 1, 0.8, 0.5), cgColorf(1, 0.2, 0, 0));
  Gradients[feRing] := lfGradient(cgColorF(1, 0.2, 0, 0.3), cgColorf(1, 0.4, 0, 0));
  Gradients[feStreaks] := lfGradient(cgColorF(1, 1, 1, 1), cgColorf(0.2, 0, 1, 0));
  Gradients[feRays] := lfGradient(cgColorF(1, 0.8, 0.5, 0.1), cgColorf(0.5, 0.2, 0, 0));
  Gradients[feSecondaries] := lfGradient(cgColorF(0, 0.2, 1, 0), cgColorf(0, 0.8, 0.2, 0.15));

end;

procedure TLFLensFlare.Render;
var
  i, j: Integer;
  rW: Single;
  a: Single;
  rnd: Single;
  v: TCGVector;
begin

  RandSeed := Seed;
  glPushAttrib(GL_ALL_ATTRIB_BITS);  // Store current OpenGL state.
  glPushMatrix;
    glTranslatef(Position.x, Position.y, Position.z);
    if feGlow in Elements then
    begin
      // Glow (a circle with transparent edges):
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glBegin(GL_TRIANGLE_FAN);
        glColor4fv(@Gradients[feGlow].CFrom);
        glVertex3f(0, 0, 0);
        glColor4fv(@Gradients[feGlow].CTo);
        for i := 0 to Resolution do
        begin
          glVertex3f(Size * cos(2*i*pi/Resolution),
                     Squeeze * Size * sin(2*i*pi/Resolution), 0);
        end;
      glEnd;
    end;
    // Ring (Three circles, the outer two are transparent):
    if feRing in Elements then
    begin
      rW := Size / 20;  // Ring width is 5% of overall flare size.
      glBlendFunc(GL_SRC_ALPHA, GL_ONE);  // Use additive blending from now on.
      glPushMatrix;
        glScalef(0.6, 0.6, 0.6);
        glBegin(GL_QUADS);
          for i := 0 to Resolution - 1 do
          begin
            glColor4fv(@Gradients[feGlow].CTo);
            glVertex3f((Size-rW) * cos(2*i*pi/Resolution),
                       Squeeze * (Size-rW) * sin(2*i*pi/Resolution), 0);
            glColor4fv(@Gradients[feRing].CFrom);
            glVertex3f(Size * cos(2*i*pi/Resolution),
                       Squeeze * Size * sin(2*i*pi/Resolution), 0);
            glVertex3f(Size * cos(2*(i+1)*pi/Resolution),
                       Squeeze * Size * sin(2*(i+1)*pi/Resolution), 0);
            glColor4fv(@Gradients[feGlow].CTo);
            glVertex3f((Size-rW) * cos(2*(i+1)*pi/Resolution),
                       Squeeze * (Size-rW) * sin(2*(i+1)*pi/Resolution), 0);

            glColor4fv(@Gradients[feRing].CFrom);
            glVertex3f(Size * cos(2*i*pi/Resolution),
                       Squeeze * Size * sin(2*i*pi/Resolution), 0);
            glVertex3f(Size * cos(2*(i+1)*pi/Resolution),
                       Squeeze * Size * sin(2*(i+1)*pi/Resolution), 0);
            glColor4fv(@Gradients[feGlow].CTo);
            glVertex3f((Size+rW) * cos(2*(i+1)*pi/Resolution),
                       Squeeze * (Size+rW) * sin(2*(i+1)*pi/Resolution), 0);
            glVertex3f((Size+rW) * cos(2*i*pi/Resolution),
                       Squeeze * (Size+rW) * sin(2*i*pi/Resolution), 0);
          end;
        glEnd;
      glPopMatrix;
    end;
    // Streaks (randomly oriented, but evenly spaced, antialiased lines from the origin):
    if feStreaks in Elements then
    begin
      glEnable(GL_LINE_SMOOTH);
      glLineWidth(StreakWidth);
      a := 2 * pi / NumStreaks;
      rnd := Random * (pi / NumStreaks);
      glBegin(GL_LINES);
        for i := 0 to NumStreaks - 1 do
        begin
          glColor4fv(@Gradients[feStreaks].CFrom);
          glVertex3f(0, 0, 0);
          glColor4fv(@Gradients[feStreaks].CTo);
          glVertex3f(1.5 * Size * cos(rnd + a*i), Squeeze * 1.5 * Size * sin(rnd + a*i), 0);
        end;
      glEnd;
    end;
    // Rays (random-length lines from the origin):
    if feRays in Elements then
    begin
      glLineWidth(1);
      glBegin(GL_LINES);
        for i := 0 to Resolution * 20 do
        begin
          glColor4fv(@Gradients[feRays].CFrom);
          glVertex3f(0, 0, 0);
          if Odd(i) then rnd := 1.5 * Random else rnd := Random;
          glColor4fv(@Gradients[feRays].CTo);
          glVertex3f(rnd * Size * cos(2*i*pi/(Resolution*20)),
                     Squeeze * rnd * Size * sin(2*i*pi/(Resolution*20)), 0);
        end;
      glEnd;
    end;
  glPopMatrix;
  if feSecondaries in Elements then
  begin
    // Huge secondary glow (a circle with a diffuse edge):
    glPushMatrix;
      glScalef(1.5, 1.5, 1.5);
      glTranslatef(Position.x, Position.y, Position.z);
      glBegin(GL_TRIANGLE_FAN);
        glColor4fv(@Gradients[feSecondaries].CFrom);
        glVertex3f(0, 0, 0);
        glColor4fv(@Gradients[feSecondaries].CTo);
        for i := 0 to Resolution do
        begin
          glVertex3f(Size * cos(2*i*pi/Resolution),
                     Squeeze * Size * sin(2*i*pi/Resolution), 0);
        end;
      glEnd;
      rW := Size / 3;
      glBegin(GL_QUADS);
        for i := 0 to Resolution - 1 do
        begin
          glColor4fv(@Gradients[feSecondaries].CTo);
          glVertex3f(Size * cos(2*i*pi/Resolution),
                     Squeeze * Size * sin(2*i*pi/Resolution), 0);
          glVertex3f(Size * cos(2*(i+1)*pi/Resolution),
                     Squeeze * Size * sin(2*(i+1)*pi/Resolution), 0);
          glColor4f(0, 0, 0, 0);
          glVertex3f((Size+rW) * cos(2*(i+1)*pi/Resolution),
                     Squeeze * (Size+rW) * sin(2*(i+1)*pi/Resolution), 0);
          glVertex3f((Size+rW) * cos(2*i*pi/Resolution),
                     Squeeze * (Size+rW) * sin(2*i*pi/Resolution), 0);
        end;
      glEnd;
    glPopMatrix;
    // Other secondaries (plain gradiented circles, like the glow):
    for j := 1 to NumSecs do
    begin
      v := Position;
      rnd := 2 * Random - 1;
      { If rnd < 0 then the secondary glow will end up on the other side of the
        origin. In this case, we can push it really far away from the flare. If
        the secondary is on the flare's side, we pull it slightly towards the
        origin to avoid it winding up in the middle of the flare. }
      if rnd < 0 then cgScale(v, 2 * rnd, 2 * rnd, 2 * rnd)
      else cgScale(v, 0.6 * rnd, 0.6 * rnd, 0.6 * rnd);
      glPushMatrix;
        glTranslatef(v.x, v.y, v.z);
        glBegin(GL_TRIANGLE_FAN);
          if j mod 3 = 0 then
          begin
            glColor4fv(@Gradients[feGlow].CFrom);
            glVertex3f(0, 0, 0);
            glColor4fv(@Gradients[feGlow].CTo);
          end
          else begin
            glColor4fv(@Gradients[feSecondaries].CFrom);
            glVertex3f(0, 0, 0);
            glColor4fv(@Gradients[feSecondaries].CTo);
          end;
          rnd := Random * Size / 4;
          for i := 0 to Resolution do
          begin
            glVertex3f(rnd * cos(2*i*pi/Resolution),
                       rnd * sin(2*i*pi/Resolution), 0);
          end;
        glEnd;
      glPopMatrix;
    end;
  end;

  glPopAttrib; // Restore the OpenGL state.

end;

end.
