unit main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DDirectDraw, DDraw;

type
  TForm1 = class(TForm)
    procedure FormCreate(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    DD: TDirectDraw;
    DDSPrimary, DDSBack, DDSOne: TDirectDrawSurface;
    DDPal: TDirectDrawPalette;
    torusColors: array [0..255] of byte;
    lastTickCount: array [0..3] of Longint;
    currentFrame: array [0..2] of Integer;

    procedure finiObjects;
    procedure doInit;
    procedure UpdateFrame(Sender: TObject; var Done: Boolean);
    function RestoreAll: Boolean;
    procedure MarkTorus;
  public
  end;

const
  szBitmap = 'ALL';
  delay: array [0..3] of Longint = (50, 78, 13, 93);
  xpos: array [0..2] of Integer = (288, 190, 416);
  ypos: array [0..2] of Integer = (128, 300, 256);

var
  Form1: TForm1;

implementation

{$R *.DFM}

(*
 * finiObjects
 *
 * finished with all objects we use; release them
 *)
procedure TForm1.finiObjects;
begin
  DD.RestoreDisplayMode;
  DD.Free;
end;


(*
 * restoreAll
 *
 * restore all lost objects
 *)
function TForm1.restoreAll: Boolean;
begin
  Result := False;
  if not DDSPrimary.Restore or not DDSOne.Restore then exit;

  DDSOne.ReloadBitmap( szBitmap );

  //Loose the old palette
  DDPal.Free;

  // create and set the palette (restart cycling from the same place)
  DDPal := DD.LoadPalette( szBitmap );

  if (DDSPrimary.PixelFormat.dwFlags and DDPF_PALETTEINDEXED8) <> 0 then
    DDSPrimary.Palette := DDPal;

  MarkTorus;

  Result := True;
end;


(*
 * doInit - do work required for every instance of the application:
 *                create the window, initialize data
 *)
procedure TForm1.doInit;
var
  ddscaps: TDDSCaps;
  ddsd: TDDSurfaceDesc;
begin
  try
    (*
     * create the main DirectDraw object
     *)
    DD := TDirectDraw.Create( nil );

    // Get exclusive mode
    DD.SetCooperativeLevel( Handle, DDSCL_EXCLUSIVE or DDSCL_FULLSCREEN );

    DD.SetDisplayMode( 640, 480, 8 );

    // Create the primary surface with 1 back buffer
    ddsd.dwSize := sizeof( ddsd );
    ddsd.dwFlags := DDSD_CAPS or DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps := DDSCAPS_PRIMARYSURFACE or
              DDSCAPS_FLIP or
              DDSCAPS_COMPLEX;
    ddsd.dwBackBufferCount := 1;
    DDSPrimary := TDirectDrawSurface.Create( DD, ddsd );

    // Get a pointer to the back buffer
    ddscaps.dwCaps := DDSCAPS_BACKBUFFER;
    DDSBack := DDSPrimary.GetAttachedSurface( ddscaps );

    // Create a Direct Draw Palette and associate it with the front buffer
    DDPal := DD.LoadPalette( szBitmap );

    DDSPrimary.Palette := DDPal;

    // Create the offscreen surface, by loading our bitmap.
    DDSOne := DD.LoadBitmap( szBitmap, 0, 0 );

    // Set the color key for this bitmap (black)
    DDSOne.SetRGBColorKey( RGB(0,0,0) );

    MarkTorus;

  except
    on Exception do
    begin
      ShowMessage( 'Direct Draw Init Failed' );
      finiObjects;
      raise;
    end;
  end;

  Application.OnIdle := UpdateFrame;
end;


procedure TForm1.MarkTorus;
var
  ddsd: TDDSurfaceDesc;
  i, x, y: Integer;
begin
  //
  // Mark the colors used in the torus frames
  //

  // First, set all colors as unused
  for i := 0 to 255 do
    torusColors[i] := 0;

  // lock the surface and scan the lower part (the torus area)
  // and remember all the index's we find.
  ddsd.dwSize := sizeof(ddsd);
  DDSOne.Lock( nil, ddsd, 0, 0 );

  // Now search through the torus frames and mark used colors
  for y := 480 to 480+383 do
    for x := 0 to 639 do
      torusColors[Byte((PChar(ddsd.lpSurface) + y*ddsd.lPitch + x)^)] := 1;

  DDSOne.Unlock;
end;


procedure TForm1.UpdateFrame(Sender: TObject; var Done: Boolean);
var
  rcRect: TRect;
  thisTickCount: Longint;
  i: Integer;
  pe: array [0..255] of TPALETTEENTRY;
begin
  Done := False;

  if DDSPrimary.IsLost then
    if not RestoreAll then exit;

  thisTickCount := GetTickCount;
  for i := 0 to 2 do
  begin
	  if((thisTickCount - lastTickCount[i]) > delay[i]) then
	  begin
	    // Move to next frame;
	    lastTickCount[i] := thisTickCount;
	    Inc( currentFrame[i] );
	    if (currentFrame[i] > 59) then
    		currentFrame[i] := 0;
    end;
  end;

  // Blit the stuff for the next frame
  rcRect := Rect( 0, 0, 640, 480 );

  try
    DDSBack.BltFast( 0, 0, DDSOne, @rcRect, DDBLTFAST_NOCOLORKEY
      or DDBLTFAST_WAIT );

    for i := 0 to 2 do
    begin
      rcRect := Rect(currentFrame[i] mod 10*64, currentFrame[i] div 10*64 + 480,
        currentFrame[i] mod 10*64 + 64, currentFrame[i] div 10*64 + 64 + 480);

      DDSBack.BltFast( xpos[i], ypos[i], DDSOne, @rcRect, DDBLTFAST_SRCCOLORKEY
        or DDBLTFAST_WAIT );
    end;

    if ((thisTickCount - lastTickCount[3]) > delay[3]) then
    begin
    	// Change the palette
    	DDPal.GetEntries( 0, 0, 256, @pe );

      for i := 1 to 255 do
        if (torusColors[i] <> 0) then
        begin
          pe[i].peRed := (pe[i].peRed + 2) mod 256;
          pe[i].peGreen := (pe[i].peGreen + 1) mod 256;
          pe[i].peBlue := (pe[i].peBlue + 3) mod 256;
        end;

      DDPal.SetEntries( 0, 0, 256, @pe );

      lastTickCount[3] := thisTickCount;
    end;

    // Flip surfaces
    DDSPrimary.Flip( nil, DDFLIP_WAIT );

  except
    on E: EDirectDrawError do
      if E.ErrorCode <> DDERR_SURFACELOST then raise;
  end;
end;


procedure TForm1.FormCreate(Sender: TObject);
var
  Cnt: Integer;
begin
  Screen.Cursor := crNone;

  for Cnt := 0 to 2 do
  begin
    lasttickCount[Cnt] := 0;
    currentFrame[Cnt] := 0;
  end;
end;

procedure TForm1.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_F10 then
    Close;
end;

procedure TForm1.FormDestroy(Sender: TObject);
begin
  finiObjects;
end;

procedure TForm1.FormShow(Sender: TObject);
begin
  if (DD = nil) then
    DoInit;
end;

end.
