unit DXFProcs;

interface

uses
  Windows, Messages, SysUtils, Dialogs;

type
  TVertex = Packed Record
    X, Y, Z: Single;
  end;
  //TPolygon is the default file-save format
  PPolygon = ^TPolygon;
  TPolygon = Packed Record
    VCount: Integer;
    Layer: Word;
    Colour: Integer;
    Vertices: Array[1..4] of TVertex;
  end;
  PPolygonArray = ^TPolygonArray;
  TPolygonArray = Array[0..0] of TPolygon;
  //TPolygonEx includes several calculated values. It is never saved directly,
  //but is always a computation of TPolygon. Usually used to add normals
  PPolygonEx = ^TPolygonEx;
  TPolygonEx = Packed Record
    VCount: Integer;
    Layer: Word;
    Colour: Integer;
    Normal: TVertex;
    Vertices: Array[1..4] of TVertex;
  end;
  PPolygonArrayEx = ^TPolygonArrayEx;
  TPolygonArrayEx = Array[0..0] of TPolygonEx;
  PModel = ^TModel;
  TModel = Packed Record
    Alloc: Integer;
    ElementCount: Integer;
    ExtentMin, ExtentMax: TVertex;
    LimitMin, LimitMax: TVertex;
    Elements: TPolygonArray;
  end;
  PModelEx = ^TModelEx;
  TModelEx = Packed Record
    Alloc: Integer;
    ElementCount: Integer;
    ExtentMin, ExtentMax: TVertex;
    LimitMin, LimitMax: TVertex;
    Elements: TPolygonArrayEx;
  end;
  TCGMFileHeader = Packed Record
    Magic: Array[0..2] of Char;
    Version: Byte;
    FirstModel: Integer;               //File Pointer to first Model, or 0 if not present
    ReserverdForFurtureExpension: Array[1..248] of Char;
  end;
  PBMPBits = ^TBMPBits;
  TBMPBits = Record
    Width: Integer;
    Height: Integer;
    Alloc: Integer;
    Bits: PByteArray;
  end;
  PRegionArray = ^TRegionArray;
  TRegionArray = Array[0..0] of TBMPBits;
  PLargeBitMap = ^TLargeBitMap;
  TLargeBitMap = Packed Record
    Width, Height: Integer;            //In regions, not pixels
    RegionCount: Integer;
    Regions: PRegionArray;
  end;

  Procedure FreeLargeBitMap(D: PLargeBitMap);
  function  IsVertexEqual(A, B: TVertex; Gap: Single): Boolean;
  function  LoadBits(FileName: PChar;  MaskFileName: PChar): TBMPBits;
  Function  LoadLargeBitMap(FileName: PChar): PLargeBitMap;
  Function  OpenCGM(FileName: String): PModel;
  Function  OpenCGMEx(FileName: String): PModelEx;
  procedure SaveAsCGM(FileName: String; Img: PModel);
  procedure SetVertex(Var V: TVertex; X, Y, Z: Single);

implementation

function IsVertexEqual(A, B: TVertex; Gap: Single): Boolean;
begin
  Result := (ABS(A.x - B.x) < Gap) AND (ABS(A.y - B.y) < Gap) AND (ABS(A.z - B.z) < Gap);
end;

function LoadBitmapFile(FileName: PChar): HBitmap;
var
  F: Integer;                 //File Handle for Windows file functions
  DC: HDC;                    //Drawing context for application
  Size, N: Longint;           //Size of bitmap, Size of color spec
  P: PBitmapInfo;             //Windows bitmap format info header
  Header: TBitmapFileHeader;  //Bitmap file header
  //Ph: TBitMapInfoHeader;
  Cs: PChar;
  PCs: Pointer;
  BytesRead: LongInt;
begin
  //This procedure is based on the old LoadBitMapFile
  //from Borland Pascal v1.5 and updated to Delphi 3
  LoadBitmapFile := 0;
  F := CreateFile(FileName, Generic_Read, File_Share_Read, nil,
    Open_Existing, 0, 0);
  if F = -1 then
    Exit;

  //Read in the Bitmap file header
  Pcs := Addr(Header);
  ReadFile(F, PCs^, SizeOf(Header), BytesRead, nil);

  //Standard Bitmaps have Header.bfType=$4d42
  //or Header.bfType = $7000

  //Read the rest of the file
  Size := GetFileSize(F, nil) - SizeOf(TBitmapFileHeader);
  GetMem(P, Size);
  ReadFile(F, P^, Size, BytesRead, nil);
  N := Header.bfOffBits - SizeOf(TBitmapFileHeader);
  PCs := P;
  Cs := PCs;
  Cs := Cs + N;

  //Create the Bitmap
  DC := GetDC(0);
  Result := CreateDIBitmap(DC, P^.bmiHeader, cbm_Init, Cs, P^, dib_RGB_Colors);

  ReleaseDC(0, DC);
  FreeMem(P, Size);
  CloseHandle(F);
end;


function LoadBits(FileName: PChar; MaskFileName: PChar): TBMPBits;
type
  TRGB = Array[1..3] of Byte;
  PRGBArray = ^TRGBArray;
  TRGBArray = Array[0..0] of TRGB;
  TRGBA = Array[1..4] of Byte;
  PRGBAArray = ^TRGBAArray;
  TRGBAArray = Array[0..0] of TRGBA;
var
  F: Integer;                          //File Handle for Windows file functions
  Size: Longint;                       //Size of bitmap, Size of color spec
  P: PBitmapInfo;                      //Windows bitmap format info header
  Header: TBitmapFileHeader;           //Bitmap file header
  Cs: PRGBArray;
  I, W: Integer;
  S: TBMPBits;
  Er: PRGBAArray;
  //Mask Info
  //MaskFileName: Array[0..4096] of Char;
  MaskSize: Longint;                   //Size of bitmap, Size of color spec
  MaskInfo: PBitmapInfo;               //Windows bitmap format info header
  MaskHeader: TBitmapFileHeader;       //Bitmap file header
  MaskBits: PRGBArray;
  Mh: THandle;
  Temp: Boolean;
begin
  //ShowMessage(FileName);
  //This procedure is based on the old LoadBitMapFile
  //from Borland Pascal v1.5 and updated to Delphi
  S.Alloc := 0;
  S.Width := 0;
  S.Height := 0;
  S.Bits := nil;
  Result := S;

  //MaskInfo := nil;
  //MaskSize := 0;
  F := CreateFile(FileName, Generic_Read, File_Share_Read, nil,
    Open_Existing, 0, 0);
  if F = -1 then
    Exit;

  //Read in the Bitmap file header
  ReadFile(F, Header, SizeOf(TBitmapFileHeader), W, nil);

  //Standard Bitmaps have Header.bfType=$4d42
  //or Header.bfType = $7000
  //Exit;

  //Read the rest of the file
  Size := GetFileSize(F, nil) - SizeOf(TBitmapFileHeader);

  GetMem(P, Size);

  ReadFile(F, P^, Size, W, nil);

  Cs := Pointer(Integer(P) + Header.bfOffBits - SizeOf(TBitmapFileHeader));
  S.Alloc := (P^.bmiHeader.biSizeImage div 3) * 4;

  GetMem(S.Bits, S.Alloc);
  S.Width := P^.bmiHeader.biWidth;
  S.Height := P^.bmiHeader.biHeight;
  CloseHandle(F);


  MaskBits := nil;
  //Mask.Alloc := 0;
  Mh := 0;
  //ShowMessage(StrPas(FileName) + ' start');
  If not (MaskFileName = nil) then
    begin
      //ShowMessage('Mask');
      //Mask := LoadBits(MaskFileName, nil);
      {FillChar(MaskFileName, 4096, 0);
      StrCopy(MaskFileName, FileName);
      //Find the final "." in FileName
      W := -1;
      For I := StrLen(MaskFileName) - 1 downto 0 do
        If MaskFileName[I] = '.' then
          begin
            W := I;
            Break;
          end;
      //Chop off extension
      If W = -1 then W := StrLen(MaskFileName);
      //StrCopy(MaskFileName + W + 1, 'mas'#0);
      MaskFileName[W + 1] := 'm';
      MaskFileName[W + 2] := 'a';
      MaskFileName[W + 3] := 's';
      MaskFileName[W + 4] := #0;}
      //Open MaskFile
      F := CreateFile(MaskFileName, Generic_Read, File_Share_Read, nil,
        Open_Existing, 0, 0);
      if not (F = Invalid_Handle_Value) then
        begin
          //Read in the file header
          ReadFile(F, MaskHeader, SizeOf(TBitmapFileHeader), W, nil);
          //Read the rest of the file
          MaskSize := GetFileSize(F, nil) - SizeOf(TBitmapFileHeader);
          Mh := GlobalAlloc(GMem_Fixed, MaskSize);
          MaskInfo := Pointer(Mh);
          If MaskInfo = nil then
            begin
              MaskBits := nil;
            end
          else
            begin
              ReadFile(F, MaskInfo^, MaskSize, W, nil);
              MaskBits := Pointer(Integer(MaskInfo) + MaskHeader.bfOffBits - SizeOf(TBitmapFileHeader));
            end;
          CloseHandle(F);
        end;
      //ShowMessage('Mask-Done');
    end;

  {If MaskBits = nil then
    ShowMessage('MaskBits = nil');
  FreeMem(P, Size);
  FreeMem(S.Bits, S.Alloc);
  If Mh <> 0 then
    GlobalFree(Mh);
  Exit;}

  Er := Pointer(S.Bits);
  //ShowMessage(StrPas(FileName) + '.A');
  Temp := AnsiPos('tropic3e', LowerCase(StrPas(FileName))) > 0;
  For I := 0 to (P^.bmiHeader.biSizeImage div 3) - 1 do
    begin
      Er^[I][1] := Cs^[I][3];
      Er^[I][2] := Cs^[I][2];
      Er^[I][3] := Cs^[I][1];
      If MaskFileName = nil then
        Er^[I][4] := $FF
      else
        If not (MaskBits = nil) then
          begin
            //Er^[I][4] := $FF
            Er^[I][4] := (MaskBits^[I][1] + MaskBits^[I][2] + MaskBits^[I][3]) div 3;
          end;
      If Temp then
        begin
          Er^[I][1] := Cs^[I][3];
          Er^[I][2] := Cs^[I][2];
          Er^[I][3] := Cs^[I][1];
          Er^[I][4] := $FF;
        end;
    end;
  //ShowMessage(StrPas(FileName) + ' near done');
  FreeMem(P, Size);

  If Mh <> 0 then
    GlobalFree(Mh);

  Result := S;
  //ShowMessage(StrPas(FileName) + ' done');
end;

{Procedure LoadBits2(FileName: PChar; HasMask: Boolean);
type
  TRGB = Array[1..3] of Byte;
  PRGBArray = ^TRGBArray;
  TRGBArray = Array[0..0] of TRGB;
  TRGBA = Array[1..4] of Byte;
  PRGBAArray = ^TRGBAArray;
  TRGBAArray = Array[0..0] of TRGBA;
var
  F: Integer;                          //File Handle for Windows file functions
  MaskSize: Longint;                       //Size of bitmap, Size of color spec
  MaskInfo: PBitmapInfo;                      //Windows bitmap format info header
  MaskHeader: TBitmapFileHeader;           //Bitmap file header
  MaskBits: PRGBArray;

  W: Integer;
  S: TBMPBits;
  //Er: PRGBAArray;

begin
  //This procedure is based on the old LoadBitMapFile
  //from Borland Pascal v1.5 and updated to Delphi
  S.Alloc := 0;
  S.Width := 0;
  S.Height := 0;
  S.Bits := nil;
  F := CreateFile(FileName, Generic_Read, File_Share_Read, nil,
    Open_Existing, 0, 0);
  if F = -1 then
    begin
    end;

  //Read in the Bitmap file header
  ReadFile(F, MaskHeader, SizeOf(MaskHeader), W, nil);

  //Standard Bitmaps have Header.bfType=$4d42
  //or Header.bfType = $7000
  //Exit;

  //Read the rest of the file
  MaskSize := GetFileSize(F, nil) - SizeOf(TBitmapFileHeader);
  GetMem(MaskInfo, MaskSize);
  ReadFile(F, MaskInfo^, MaskSize, W, nil);
  MaskBits := Pointer(Integer(MaskInfo) + MaskHeader.bfOffBits - SizeOf(TBitmapFileHeader));
  S.Alloc := (MaskInfo^.bmiHeader.biSizeImage div 3) * 4;
  GetMem(S.Bits, S.Alloc);
  S.Width := MaskInfo^.bmiHeader.biWidth;
  S.Height := MaskInfo^.bmiHeader.biHeight;

  CloseHandle(F);
  FreeMem(MaskInfo, MaskSize);

end;}

Function bmAlignDouble(Size: LongInt): LongInt;
begin
  Result := (Size + 31) div 32 * 4;
end;

Procedure FreeLargeBitMap(D: PLargeBitMap);
var
  I: Integer;
begin
  If D = nil then
    Exit;
  For I := 0 to D^.RegionCount - 1 do
    FreeMem(D^.Regions[I].Bits, D^.Regions[I].Alloc);
  FreeMem(D^.Regions, D^.RegionCount * SizeOf(TBMPBits));
  FreeMem(D, SizeOf(TLargeBitMap));
end;

Function LoadLargeBitMap(FileName: PChar): PLargeBitMap;
type
  TRGB = Array[1..3] of Byte;
  PRGBArray = ^TRGBArray;
  TRGBArray = Array[0..0] of TRGB;
var
  BIP: PBitMapInfo;
  BitBuf: PByteArray;
  DataSize: Integer;
  Dc, MemL, MemS: THandle;
  E: Integer;
  Er, Fr: PRGBArray;
  I: Integer;
  MainBMP, SmallBMP: THandle;
  MainBMPData: TBitMap;
  Rs: PLargeBitMap;
  Rx, Ry, Ri: Integer;
  A: String;
begin
  Result := nil;
  MainBMP := LoadBitmapFile(FileName);
  If MainBMP = 0 then
    Exit;

  GetObject(MainBMP, SizeOf(TBitmap), @MainBMPData);
  GetMem(Rs, SizeOf(TLargeBitMap));
  Rs^.Width := MainBMPData.bmWidth div 256;
  Rs^.Height := MainBMPData.bmHeight div 256;
  Rs^.RegionCount := Rs^.Width * Rs^.Height;
  GetMem(Rs^.Regions, Rs^.RegionCount * SizeOf(TBMPBits));
  //Create device contexts
  Dc := GetDC(0);
  MemL := CreateCompatibleDC(DC);
  MemS := CreateCompatibleDC(DC);
  //Create a Smaller BitMap that will be used to select regions in the larger
  SmallBMP := CreateCompatibleBitMap(Dc, 256, 256);
  //Initialize the buffers pointing at SmallBMP
  GetMem(BIP, SizeOf(TBitMapInfo) + 10);
  Bip^.bmiHeader.biSize := SizeOf(TBitMapInfoHeader);
  E := GetDiBits(Dc, SmallBMP, 0, 256, nil, BIP^, dib_RGB_Colors);
  If E = 0 then
    begin
      //error!
      BIP^.bmiHeader.biWidth := 256;
      BIP^.bmiHeader.biHeight := 256;
    end;
  With BIP^.bmiHeader do
    begin
      DataSize := bmAlignDouble(biWidth * 24) * biHeight;
      biPlanes := 1;
      biBitCount := 24;
      biCompression := 0;
      biSizeImage := DataSize;
    end;
  //Allocate Buffer
  GetMem(BitBuf, DataSize);

  SelectObject(MemL, MainBMP);
  SelectObject(MemS, SmallBMP);
  A := '';
  //Load the pieces
  For Rx := 0 to Rs^.Width - 1 do
    For Ry := 0 to Rs^.Height - 1 do
      begin
        //Copy this piece into the buffer
        BitBLT(MemS, 0, 0, 256, 256, MemL, Rx * 256, Ry * 256, SrcCopy);
        //Get the Bits
        GetDiBits(MemS, SmallBMP, 0, 256, BitBuf, BIP^, dib_RGB_Colors);
        {If (Length(A) = 0) then
          begin
            //A := 'GetDIB Error ' + IntToStr(GetLastError);
            A := 'ScanLines: ' + IntToStr(I);
            MessageBox(0, PChar(A), '', 0);
          end;}
        //Decide where this data will be saved
        Ri := Rx + Ry * Rs^.Width;
        //Save it
        Rs^.Regions[Ri].Width := 256;
        Rs^.Regions[Ri].Height := 256;
        Rs^.Regions[Ri].Alloc := 256 * 256 * 3;
        GetMem(Rs^.Regions[Ri].Bits, Rs^.Regions[Ri].Alloc);
        Er := Pointer(Rs^.Regions[Ri].Bits);
        Fr := Pointer(BitBuf);
        For I := 0 to 256 * 256 - 1 do
          begin
            Er^[I][1] := Fr^[I][3];
            Er^[I][2] := Fr^[I][2];
            Er^[I][3] := Fr^[I][1];
            {Er^[I][1] := 0;
            Er^[I][2] := 0;
            Er^[I][3] := 0;}
          end;
        //Move(BitBuf^, Rs^.Regions[Ri].Bits^, Rs^.Regions[Ri].Alloc);
        //If Length(A) = 0 then
        {If Rs^.Regions[Ri].Bits = nil then
          begin
            //A := 'DSize = ' + IntToStr(DataSize);
            //A := 'Alloc = ' + IntToStr(Rs^.Regions[Ri].Alloc);
            //A := 'Width = ' + IntToStr(Rs^.Width) + ' Height = ' + IntToStr(Rs^.Height);
            A := 'Null! @' + IntToStr(Ry) + ',' + IntToStr(Ry);
            MessageBox(0, PChar(A), '', 0);
          end;}
      end;

  FreeMem(BitBuf, DataSize);
  FreeMem(BIP, SizeOf(TBitMapInfo) + 10);
  DeleteDC(MemL);
  DeleteDC(MemS);
  ReleaseDC(0, DC);
  DeleteObject(MainBMP);
  DeleteObject(SmallBMP);

  Result := Rs;
  //MessageBox(0, 'Z', '', 0);
end;

procedure SaveAsCGM(FileName: String; Img: PModel);
var
  Fh: THandle;
  Head: TCGMFileHeader;
  W: DWord;
begin
  Fh := CreateFile(PChar(FileName), Generic_Read OR Generic_Write,
    File_Share_Read OR File_Share_Write, nil, Create_Always, 0, 0);
  FillChar(Head, SizeOf(TCGMFileHeader), 0);
  StrCopy(Head.Magic, 'CGM');
  Head.Version := 1;
  Head.FirstModel := SizeOf(TCGMFileHeader);
  WriteFile(Fh, Head, SizeOf(TCGMFileHeader), W, nil);
  WriteFile(Fh, Img^, Img^.Alloc, W, nil);
  CloseHandle(Fh);
end;

Function OpenCGM(FileName: String): PModel;
var
  Fh: THandle;
  Head: TCGMFileHeader;
  W: DWord;
  Img: PModel;
  G: TModel;
begin
  Img := nil;
  Result := Img;
  Fh := CreateFile(PChar(FileName), Generic_Read OR Generic_Write,
    File_Share_Read OR File_Share_Write, nil, Open_Existing, 0, 0);
  If Fh = Invalid_Handle_Value then
    Exit;
  ReadFile(Fh, Head, SizeOf(TCGMFileHeader), W, nil);
  If Head.FirstModel > 0 then
    begin
      SetFilePointer(Fh, Head.FirstModel, nil, File_Begin);
      ReadFile(Fh, G, SizeOf(TModel), W, nil);
      GetMem(Img, G.Alloc);
      SetFilePointer(Fh, Head.FirstModel, nil, File_Begin);
      ReadFile(Fh, Img^, G.Alloc, W, nil);
    end;
  Result := Img;
  CloseHandle(Fh);
end;

Function OpenCGMEx(FileName: String): PModelEx;
var
  Img: PModel;
  ImgEx: PModelEx;
  I, J, Nw: Integer;
  A, B: TVertex;

  procedure Normalize(var v : TVertex);
  var d : Single;
  begin
    d := sqrt(v.x*v.x+v.y*v.y+v.z*v.z);
    if (d = 0.0) then
      begin
        // Error
        exit;
      end;
    v.x := v.x / d;
    v.y := v.y / d;
    v.z := v.z / d;
  end;

begin
  Img := OpenCGM(FileName);
  If Img = nil then
    begin
      Result := nil;
      Exit;
    end;
  Nw := SizeOf(TModelEx) + Img^.ElementCount * SizeOf(TPolygonEx);
  GetMem(ImgEx, Nw);
  //FillChar(ImgEx, Nw, 0);
  ImgEx^.ElementCount := Img^.ElementCount;
  ImgEx^.ExtentMin := Img^.ExtentMin;
  ImgEx^.ExtentMax := Img^.ExtentMax;
  ImgEx^.LimitMin := Img^.LimitMin;
  ImgEx^.LimitMax := Img^.LimitMax;
  ImgEx^.Alloc := Nw;
  For I := 0 to ImgEx^.ElementCount - 1 do
    begin
      ImgEx^.Elements[I].VCount := Img^.Elements[I].VCount;
      ImgEx^.Elements[I].Layer := Img^.Elements[I].Layer;
      ImgEx^.Elements[I].Colour := Img^.Elements[I].Colour;
      For J := 1 to 4 do
        ImgEx^.Elements[I].Vertices[J] := Img^.Elements[I].Vertices[J];
      {If (ImgEx^.Elements[I].Vertices[1].X = 0) AND (ImgEx^.Elements[I].Vertices[1].Y = 0) then
        ImgEx^.Elements[I].Vertices[1] := ImgEx^.Elements[I].Vertices[4];
      If (ImgEx^.Elements[I].Vertices[2].X = 0) AND (ImgEx^.Elements[I].Vertices[2].Y = 0) then
        ImgEx^.Elements[I].Vertices[2] := ImgEx^.Elements[I].Vertices[4];
      If (ImgEx^.Elements[I].Vertices[3].X = 0) AND (ImgEx^.Elements[I].Vertices[2].Y = 0) then
        ImgEx^.Elements[I].Vertices[3] := ImgEx^.Elements[I].Vertices[4];}
      //Compute Normal using cross multiplication
      A.X := ImgEx^.Elements[I].Vertices[1].X - ImgEx^.Elements[I].Vertices[2].X;
      A.Y := ImgEx^.Elements[I].Vertices[1].Y - ImgEx^.Elements[I].Vertices[2].Y;
      A.Z := ImgEx^.Elements[I].Vertices[1].Z - ImgEx^.Elements[I].Vertices[2].Z;
      B.X := ImgEx^.Elements[I].Vertices[1].X - ImgEx^.Elements[I].Vertices[3].X;
      B.Y := ImgEx^.Elements[I].Vertices[1].Y - ImgEx^.Elements[I].Vertices[3].Y;
      B.Z := ImgEx^.Elements[I].Vertices[1].Z - ImgEx^.Elements[I].Vertices[3].Z;
      ImgEx^.Elements[I].Normal.X := A.Y * B.Z - A.Z * B.Y;
      ImgEx^.Elements[I].Normal.Y := A.Z * B.X - A.X * B.Z;
      ImgEx^.Elements[I].Normal.Z := A.X * B.Y - A.Y * B.X;
      //Normalize(ImgEx^.Elements[I].Normal);
      {While ABS(ImgEx^.Elements[I].Normal.X) > 1 do
        ImgEx^.Elements[I].Normal.X := ImgEx^.Elements[I].Normal.X / 2;
      While ABS(ImgEx^.Elements[I].Normal.Y) > 1 do
        ImgEx^.Elements[I].Normal.Y := ImgEx^.Elements[I].Normal.Y / 2;
      While ABS(ImgEx^.Elements[I].Normal.Z) > 1 do
        ImgEx^.Elements[I].Normal.Z := ImgEx^.Elements[I].Normal.Z / 2;}
      {ImgEx^.Elements[I].Normal.X := ImgEx^.Elements[I].Normal.X / 500;
      ImgEx^.Elements[I].Normal.Y := ImgEx^.Elements[I].Normal.Y / 500;
      ImgEx^.Elements[I].Normal.Z := ImgEx^.Elements[I].Normal.Z / 500;}
      {ImgEx^.Elements[I].Normal.X := 0;
      ImgEx^.Elements[I].Normal.Y := 0;
      ImgEx^.Elements[I].Normal.Z := 1;}
      {If ImgEx^.Elements[I].Normal.X > 10 then
        ImgEx^.Elements[I].Normal.X := 1
      else
        If ImgEx^.Elements[I].Normal.X < -10 then
          ImgEx^.Elements[I].Normal.X := -1
        else
          ImgEx^.Elements[I].Normal.X := 0;
      If ImgEx^.Elements[I].Normal.Y > 10 then
        ImgEx^.Elements[I].Normal.Y := 1
      else
        If ImgEx^.Elements[I].Normal.Y < -10 then
          ImgEx^.Elements[I].Normal.Y := -1
        else
          ImgEx^.Elements[I].Normal.Y := 0;
      If ImgEx^.Elements[I].Normal.Z > 10 then
        ImgEx^.Elements[I].Normal.Z := 1
      else
        If ImgEx^.Elements[I].Normal.Z < -10 then
          ImgEx^.Elements[I].Normal.Z := -1
        else
          ImgEx^.Elements[I].Normal.Z := 0;}
    end;

  Result := ImgEx;
  FreeMem(Img, Img^.Alloc);
end;

procedure SetVertex(Var V: TVertex; X, Y, Z: Single);
begin
  V.X := X;
  V.Y := Y;
  V.Z := Z;
end;

end.

