Unit uSimpleTable;

interface

{TSimpleTable. Database-interpretation of the Datamodule
  By: Keneto
  Date: Mar/2005
  }

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, Db, DBCtrls;  

type
  //TSimpleTable
  PDataBuffer = ^TDataBuffer;
  TDataBuffer = Record
    RecNo: Integer;
    IsNew: Boolean;
  end;
  //For Index
  TEntry = Record
    MyID: Integer;
    DetailID: Integer;
    FileIndex: Integer;
    SortData: Integer;
  end;
  TCharD = Array[0..3] of char;
  PIndex = ^TIndex;
  TIndex = Array[0..0] of TEntry;

const
  EntrySize = SizeOf(TEntry);

type
  TSimpleTable = class(TDataSet)
    private
      AutoInc: PInteger;               //Pointed by descendant to the field that must be autonumbered
      AutoIncSource: PInteger;         //Pointed by descendant to field in Datamodule that tracks
      Children: TList;
      fDataSize: Integer;
      fHData: THandle;
      fHSaved: THandle;                //DataFile to be used during open
      fMasterTable: TSimpleTable;
      fRecBufSize: LongInt;
      fRecordCount: Integer;
      fRecNo: LongInt;
      fSortField: Integer;
      Index, SubIndex: PIndex;
      IndexAlloc: Integer;             //Number of entries allocated in the index. May be more than RecordCount
      LoadDataPtr: Pointer;            //Buffer where data will be loaded
      MasterField: Integer;
      NeedsIndex: Boolean;             //Internal use only to alert self to build index
      fNeedsSubIndex: Boolean;         //Internal use only
      SubIndexAlloc: Integer;          //Number of entries allocated in the subindex
      SubRecordCount: Integer;         //Used only when a child / detail table. When master use file
      Procedure AddChild(AValue: TSimpleTable); //Master-Detail
      Function  AllowEntry: Boolean; virtual;
      Procedure BuildIndex;
      Procedure BuildSubIndex;
      Procedure CatchEntry(var E: TEntry); virtual; //Master-Detail children must catch this and fill the DetailID structure
      Procedure DeleteChild(AValue: TSimpleTable); //Master-Detail
      procedure GetData(Field: TField; Buffer: Pointer); virtual;
      Procedure LoadData(AIndex: Integer); virtual;
      Procedure MasterChangedProc; virtual;
      Function  NeedsSubIndex: Boolean;
      procedure SetData(Field: TField; Buffer: Pointer); virtual;
      Procedure SetMasterTable(AValue: TSimpleTable);
      Procedure SetSortField(AIndex: Integer);
    protected
      function  GetRecNo: Integer; override;
      procedure InternalCancel; override;
      procedure InternalEdit; override;
      procedure InternalRefresh; override;
      procedure SetFiltered(Value: Boolean); override;
      procedure SetRecNo(Value: Integer); override;
    protected
      function  AllocRecordBuffer: PChar; override;
      Procedure CheckBuffer(Buffer: PChar);
      procedure FreeRecordBuffer(var Buffer: PChar); override;
      procedure GetBookmarkData(Buffer: PChar; Data: Pointer); override;
      function  GetBookmarkFlag(Buffer: PChar): TBookmarkFlag; override;
      function  GetFieldData(Field: TField; Buffer: Pointer): Boolean; override;
      function  GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean): TGetResult; override;
      function  GetRecordCount: Longint; override;
      function  GetRecordSize: Word; override;
      procedure InternalAddRecord(Buffer: Pointer; Append: Boolean); override;
      procedure InternalClose; override;
      procedure InternalDelete; override;
      procedure InternalFirst; override;
      procedure InternalGotoBookmark(Bookmark: Pointer); override;
      procedure InternalHandleException; override;
      procedure InternalInitRecord(Buffer: PChar); override;
      procedure InternalLast; override;
      procedure InternalOpen; override;
      procedure InternalPost; override;
      procedure InternalSetToRecord(Buffer: PChar); override;
      function  IsCursorOpen: Boolean; override;
      procedure SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag); override;
      procedure SetBookmarkData(Buffer: PChar; Data: Pointer); override;
      procedure SetFieldData(Field: TField; Buffer: Pointer); override;
    public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      Procedure CheckActiveBuffer;
      Procedure DeleteAll; virtual;
      procedure DoAfterCancel; override;
      procedure DoAfterScroll; override;
      function  GetByID(ID: Integer; Dest: Pointer): Boolean;
      function  GetCurrentRecord(Buffer: PChar): Boolean; override;
      Function  InternalGetName: String;
      Property  ActiveRecord;
      Property  BufferCount;
      Property  Buffers;
      Property  RecNo: LongInt read fRecNo write SetRecNo;
    published
      Property MasterTable: TSimpleTable read fMasterTable write SetMasterTable;
      Property SortField: Integer read fSortField write SetSortField;
      Property Active;
      Property Filter;
      Property Filtered;
      property BeforeOpen;
      property AfterOpen;
      property BeforeClose;
      property AfterClose;
      property BeforeInsert;
      property AfterInsert;
      property BeforeEdit;
      property AfterEdit;
      property BeforePost;
      property AfterPost;
      property BeforeCancel;
      property AfterCancel;
      property BeforeDelete;
      property AfterDelete;
      property BeforeScroll;
      property AfterScroll;
      property OnCalcFields;
      property OnDeleteError;
      property OnEditError;
      property OnFilterRecord;
      property OnNewRecord;
      property OnPostError;
    end;


  TSentence = Packed Record
    ID: Integer;
    TextID: Integer;
    CategoryID: Integer;               //Not used
    Text: String[80];
    Len: Integer;                      //If Len > 80 then use BLOB
    FHigh: DWord;                      //HighDWord of FileIndex into sDataFile
    FLow: DWord;                       //LowDWord of FileIndex into sDataFile
    HitsLatest: Integer;
    HitsBefore: Integer;
    AllocLen: Integer;
    ReservdForFutureExpansion: Array[1..139] of Byte;
  end; //SizeOf(TSentence) = 256

  TSentenceTable = class(TSimpleTable)
    private
      Data: TSentence;
      SenBlob: THandle;  //For BLOB Data
      procedure GetData(Field: TField; Buffer: Pointer); override;
      procedure SetData(Field: TField; Buffer: Pointer); override;
    protected
      procedure InternalInitFieldDefs; override;
    public
      constructor Create(AOwner: TComponent); override;
      Procedure DeleteAll; override;
  end;

implementation

type
  TIDNumberList = Packed Record
    SentenceID: Integer;
  end;
  TDM = class(TPersistent)
    private
      IDNumberList: TIDNumberList;
    protected
      Procedure LoadSettings;
      Procedure SaveSettings;
  end;

var
  DM: TDM;

Procedure TDM.LoadSettings;
begin
  //Load the ID numbers from a file.
  //This guarantees that an ID number won't be re-used
end;

Procedure TDM.SaveSettings;
begin
  //Save the ID numbers to a file.
  //This guarantees that an ID number won't be re-used
end;

constructor TSimpleTable.Create(AOwner: TComponent);
begin
  fRecBufSize := SizeOf(TDataBuffer);
  fRecNo := -1;
  Inherited Create(AOwner);
  fMasterTable := nil;
  Children := TList.Create;
  fHData := 0;
  fSortField := -1;
  MasterField := -1;
  IndexAlloc := 0;
  SubRecordCount := 0;
  SubIndexAlloc := 0;
  NeedsIndex := True;
  fNeedsSubIndex := True;
  AutoInc := nil;
  AutoIncSource := nil;
end;

Procedure TSimpleTable.AddChild(AValue: TSimpleTable);
begin
  If Children.IndexOf(AValue) = -1 then
    Children.Add(AValue);
end;

Function TSimpleTable.AllowEntry: Boolean;
begin
  Result := True;
end;

function  TSimpleTable.AllocRecordBuffer: PChar;
var
  Cs: PChar;
begin
  GetMem(Cs, SizeOf(TDataBuffer));
  PDataBuffer(Cs)^.IsNew := False;
  Result := Cs;
end;

Procedure TSimpleTable.BuildIndex;
type
  PMyPtr = ^TMyPtr;
  TMyPtr = Packed Record
    ID: Integer;
    Other: Array[0..0] of byte;
  end;
var
  I, X: Integer;
  MyPtr: PMyPtr;
  W: DWord;
  Buffer: Array[0..4096] of Char;
  Busy: Boolean;
  T: TEntry;
begin
  If not NeedsIndex then
    begin
      If NeedsSubIndex then
        BuildSubIndex;
      Exit;
    end;
  X := GetFileSize(fhData, nil) div fDataSize;
  If X > IndexAlloc then
    begin
      //Re-Allocate the index. Give it around 10 extra entries to
      //allow for rapid re-sizing on-demand. Discard the older one
      If IndexAlloc > 0 then
        FreeMem(Index, IndexAlloc * EntrySize);
      IndexAlloc := X + 10;
      GetMem(Index, IndexAlloc * EntrySize);
    end;
  //Blank out and resident Garbage / old data
  FillChar(Index^, IndexAlloc * EntrySize, 0);
  MyPtr := LoadDataPtr;
  fRecordCount := 0;
  For I := 0 to X - 1 do
    begin
      SetFilePointer(fHData, I * fDataSize, nil, File_begin);
      ReadFile(fHData, LoadDataPtr^, fDataSize, W, nil);
      If AllowEntry then
        begin
          Index^[fRecordCount].MyID := MyPtr^.ID;
          Index^[fRecordCount].FileIndex := I * fDataSize;
          //Allow descendant to fill in any extra index data
          CatchEntry(Index^[fRecordCount]);
          //Copy some of the data over to the index's sort data
          If SortField > -1 then
            begin
              GetData(Fields[SortField], @Buffer);
              Move(Buffer, Index^[fRecordCount].SortData, 4);
            end;
          Inc(fRecordCount);
        end;
    end;
  NeedsIndex := False;

  //Bubble Sort the index. This is highly inefficient... however
  //the only table that uses sorts is the Synomyn table, which only
  //contains a few hundred records.
  If SortField > -1 then
    begin
      Busy := True;
      While Busy do
        begin
          Busy := False;
          For I := 0 to X - 2 do
            If (Fields[SortField].DataType = ftInteger) AND (Index^[I].SortData > Index^[I + 1].SortData) OR
               (Fields[SortField].DataType = ftString) AND (LowerCase(TCharD(Index^[I].SortData)) > LowerCase(TCharD(Index^[I + 1].SortData))) then
              begin
                T := Index^[I];
                Index^[I] := Index^[I + 1];
                Index^[I + 1] := T;
                Busy := True;
              end;
        end;
    end;

  If NeedsSubIndex then
    BuildSubIndex;
end;

Procedure TSimpleTable.BuildSubIndex;
type
  PMyPtr = ^TMyPtr;
  TMyPtr = Packed Record
    ID: Integer;
    Other: Array[0..0] of byte;
  end;

  Procedure ReAllocateSubIndex;
  var
    Temp: Pointer;
    TempAlloc: Integer;
  begin
    //Re-Allocate the index. Give it around 10 extra entries to
    //allow for rapid re-sizing on-demand. Discard the older one
    TempAlloc := SubIndexAlloc + 50;
    GetMem(Temp, TempAlloc * EntrySize);
    If SubIndexAlloc > 0 then
      begin
        Move(SubIndex^, Temp^, SubIndexAlloc * EntrySize);
        FreeMem(SubIndex, SubIndexAlloc * EntrySize);
      end;
    SubIndex := Temp;
    SubIndexAlloc := TempAlloc;
  end;

var
  I, X: Integer;
begin
  If MasterField = -1 then
    Exit;
  If NeedsIndex then
    BuildIndex;
  If not NeedsSubIndex then
    Exit;
  SubRecordCount := 0;
  X := GetFileSize(fhData, nil) div fDataSize;
  If SubIndexAlloc < 10 then
    ReAllocateSubIndex;
  MasterTable.CheckActiveBuffer;
  For I := 0 to X - 1 do
    If Index^[I].DetailID = MasterTable.AutoInc^ then
      begin
        SubIndex[SubRecordCount] := Index^[I];
        Inc(SubRecordCount);
        If SubRecordCount >= SubIndexAlloc - 1 then
          ReAllocateSubIndex;
      end;
  fNeedsSubIndex := False;
end;

Procedure TSimpleTable.CatchEntry(var E: TEntry);
begin
  E.DetailID := -1;
end;

Procedure TSimpleTable.CheckActiveBuffer;
begin
  CheckBuffer(ActiveBuffer);
end;

Procedure TSimpleTable.CheckBuffer(Buffer: PChar);
var
  OldRec: Integer;
begin
  If Buffer = nil then
    Exit;
  If NeedsIndex then
    BuildIndex;
  OldRec := fRecNo;
  fRecNo := PDataBuffer(Buffer)^.RecNo;
  if not (OldRec = fRecNo) then
    If not PDataBuffer(Buffer)^.IsNew then
      LoadData(fRecNo);
end;

Procedure TSimpleTable.DeleteChild(AValue: TSimpleTable);
var
  X: Integer;
begin
  X := Children.IndexOf(AValue);
  If X > -1 then
    Children.Delete(X);
end;

destructor TSimpleTable.Destroy;
var
  I: Integer;
begin
  Inherited Destroy;
  If Children.Count > 0 then
    For I := 0 to Children.Count - 1 do
      TSimpleTable(Children[I]).MasterTable := nil;
  Children.Free;
  If IndexAlloc > 0 then
    begin
      FreeMem(Index, IndexAlloc * EntrySize);
      IndexAlloc := 0;
    end;
  If SubIndexAlloc > 0 then
    begin
      FreeMem(SubIndex, SubIndexAlloc * EntrySize);
      SubIndexAlloc := 0;
    end;
end;

Procedure TSimpleTable.DeleteAll;
begin
  SetFilePointer(fHData, 0, nil, File_begin);
  SetEndOfFile(fhData);
  If not (AutoIncSource = nil) then
    begin
      AutoIncSource^ := 1;
      DM.SaveSettings;
    end;
  Refresh;
end;

procedure TSimpleTable.DoAfterCancel;
begin
  Inherited DoAfterCancel;
  Prior;
  Refresh;
end;

procedure TSimpleTable.DoAfterScroll;
var
  I: Integer;
begin
  Inherited DoAfterScroll;
  For I := 0 to Children.Count - 1 do
    TSimpleTable(Children[I]).MasterChangedProc;
end;

procedure TSimpleTable.FreeRecordBuffer(var Buffer: PChar);
begin
  try
    If not (Buffer = nil) then
      FreeMem(Buffer, SizeOf(TDataBuffer));
    Buffer := nil;
  except
    On Exception do
      begin end;
  end;
end;

procedure TSimpleTable.GetBookmarkData(Buffer: PChar; Data: Pointer);
begin
end;

function  TSimpleTable.GetBookmarkFlag(Buffer: PChar): TBookmarkFlag;
begin
  Result := bfCurrent;
end;

function TSimpleTable.GetByID(ID: Integer; Dest: Pointer): Boolean;
var
  I: Integer;
  W: DWord;
begin
  //Use the main index to get the current data
  Result := False;
  For I := 0 to RecordCount - 1 do
    If Index^[I].MyID = ID then
      begin
        Result := True;
        SetFilePointer(fHData, Index^[I].FileIndex, nil, File_Begin);
        ReadFile(fHData, Dest^, fDataSize, W, nil);
      end;
end;

function TSimpleTable.GetCurrentRecord(Buffer: PChar): Boolean;
begin
  Result := not (Buffer = nil);
  PDataBuffer(Buffer)^.RecNo := fRecNo;
end;

procedure TSimpleTable.GetData(Field: TField; Buffer: Pointer);
begin
end;

function TSimpleTable.GetFieldData(Field: TField; Buffer: Pointer): Boolean;
begin
  Result := False;
  If PDataBuffer(ActiveBuffer)^.IsNew then
    begin
      If Field.Index = 0 then
        begin
          PInteger(Buffer)^ := AutoIncSource^;
          Result := True;
        end;
      Case Field.DataType of
        ftInteger: PInteger(Buffer)^ := 0;
        ftString: StrCopy(Buffer, '');
      end;
      Exit;
    end;
  CheckActiveBuffer;
  GetData(Field, Buffer);
  Result := True;
end;

function TSimpleTable.GetRecNo: Integer;
begin
  Result := fRecNo;
end;

function TSimpleTable.GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean): TGetResult;
begin
  Result := grOk;
  If PDataBuffer(Buffer)^.IsNew then
    begin
      FillChar(LoadDataPtr^, fDataSize, 0);
      Result := grOk;
      Exit;
    end;
  case GetMode of
    gmCurrent:
      begin
        //CheckActiveBuffer;
        If RecordCount > 0 then
          LoadData(fRecNo)
        else
          begin
            FillChar(LoadDataPtr^, fDataSize, 0);
            Result := grEOF;
          end;
      end;
    gmNext:
      If fRecNo >= RecordCount - 1 then
        Result := grEOF
      else
        begin
          Inc(fRecNo);
          LoadData(fRecNo);
        end;
    gmPrior:
      If fRecNo < 1 then
        Result := grBOF
      else
        begin
          Dec(fRecNo);
          LoadData(fRecNo);
        end;
  end;
  If not (Buffer = nil) then
    If Result = grOk then
      PDataBuffer(Buffer)^.RecNo := fRecNo;
end;

function TSimpleTable.GetRecordCount: Longint;
begin
  If fMasterTable = nil then
    begin
      If Filtered then
        Result := fRecordCount
      else
        Result := GetFileSize(fhData, nil) div fDataSize
    end
  else
    begin
      BuildSubIndex;
      Result := SubRecordCount;
    end;
end;

function TSimpleTable.GetRecordSize: Word;
begin
  Result := fDataSize;
end;

procedure TSimpleTable.InternalAddRecord(Buffer: Pointer; Append: Boolean);
begin
end;

procedure TSimpleTable.InternalCancel;
begin
end;

procedure TSimpleTable.InternalEdit;
begin
end;

procedure TSimpleTable.InternalRefresh;
begin
  NeedsIndex := True;
  fNeedsSubIndex := True;
  BuildIndex;
end;

procedure TSimpleTable.InternalClose;
begin
  //This needs to close the RecordSet
  BindFields(False);
  if DefaultFields then
    DestroyFields;
end;

procedure TSimpleTable.InternalDelete;
var
  I: Integer;
  Temp: Pointer;
  W: DWord;
  _First, _Last: Integer;
begin
  CheckActiveBuffer;
  //Make buffer
  GetMem(Temp, fDataSize);
  //Set First & Last record to move
  If fMasterTable = nil then
    _First := Index[fRecNo].FileIndex div fDataSize
  else
    _First := SubIndex[fRecNo].FileIndex div fDataSize;
  _Last := GetFileSize(fhData, nil) div fDataSize;
  //Copy the records over starting from one after record to be deleted
  For I := _First + 1 to _Last do
    begin
      SetFilePointer(fhData, I * fDataSize, nil, File_Begin);
      ReadFile(fhData, Temp^, fDataSize, W, nil);
      SetFilePointer(fhData, (I - 1) * fDataSize, nil, File_Begin);
      WriteFile(fhData, Temp^, fDataSize, W, nil);
    end;
  //New file end
  SetFilePointer(fhData, -fDataSize, nil, File_End);
  SetEndOfFile(fhData);
  //Free buffer
  FreeMem(Temp, fDataSize);

  If fRecNo >= RecordCount then
    fRecNo := RecordCount - 1;

  NeedsIndex := True;
  fNeedsSubIndex := True;
  BuildIndex;
end;

procedure TSimpleTable.InternalFirst;
begin
  fRecNo := -1;
end;

Function TSimpleTable.InternalGetName: String ;
begin
  Result := Name;
end;

procedure TSimpleTable.InternalGotoBookmark(Bookmark: Pointer);
begin
end;

procedure TSimpleTable.InternalHandleException;
begin
end;

procedure TSimpleTable.InternalInitRecord(Buffer: PChar);
begin
  If not (Buffer = nil) then
    begin
      PDataBuffer(Buffer)^.IsNew := True;
      PDataBuffer(Buffer)^.RecNo := RecordCount;
      FillChar(LoadDataPtr^, fDataSize, 0);
    end;
end;

procedure TSimpleTable.InternalLast;
begin
  fRecNo := RecordCount; //-1;
end;

procedure TSimpleTable.InternalOpen;
begin
  InternalInitFieldDefs;
  if DefaultFields then
    CreateFields;
  BindFields(True);
  fHData := fHSaved;
  BuildIndex;
end;

procedure TSimpleTable.InternalPost;
begin
end;

procedure TSimpleTable.InternalSetToRecord(Buffer: PChar);
begin
  CheckBuffer(Buffer);
end;

function  TSimpleTable.IsCursorOpen: Boolean;
begin
  Result := not (fHData = 0);
end;

Procedure TSimpleTable.LoadData(AIndex: Integer);
var
  W: DWord;
  E: Integer;
begin
  FillChar(LoadDataPtr^, fDataSize, 0);
  If (AIndex >= 0) AND (AIndex < RecordCount) then
    begin
      If NeedsIndex then
        BuildIndex;
      If fMasterTable = nil then
        E := Index[AIndex].FileIndex
      else
        E := SubIndex[AIndex].FileIndex;
      SetFilePointer(fHData, E, nil, File_begin);
      ReadFile(fHData, LoadDataPtr^, fDataSize, W, nil);
    end;
end;

Procedure TSimpleTable.MasterChangedProc;
begin
  If MasterField = -1 then
    Exit;
  Refresh;
end;

Function TSimpleTable.NeedsSubIndex: Boolean;
begin
  Result := fNeedsSubIndex AND not (fMasterTable = nil);
end;

procedure TSimpleTable.SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag);
begin
end;

procedure TSimpleTable.SetBookmarkData(Buffer: PChar; Data: Pointer);
begin
end;

procedure TSimpleTable.SetFiltered(Value: Boolean);
begin
  Inherited SetFiltered(Value);
  Refresh;
end;

procedure TSimpleTable.SetFieldData(Field: TField; Buffer: Pointer);
var
  W: DWord;
  I, E: Integer;
begin
  If Buffer = nil then
    Exit;
  If NeedsIndex then
    BuildIndex;
  CheckActiveBuffer;
  If PDataBuffer(ActiveBuffer)^.IsNew then
    FillChar(LoadDataPtr^, fDataSize, 0);
  //Call Descendant to Fill the DataBuffer
  SetData(Field, Buffer);
  If PDataBuffer(ActiveBuffer)^.IsNew then
    begin
      SetFilePointer(fHData, 0, nil, File_End);
      If not (AutoInc = nil) then
        begin
          AutoInc^ := AutoIncSource^;
          Inc(AutoIncSource^);
          DM.SaveSettings;
        end;
      PDataBuffer(ActiveBuffer)^.RecNo := RecordCount;
    end
  else
    begin
      If fMasterTable = nil then
        E := Index[fRecNo].FileIndex
      else
        E := SubIndex[fRecNo].FileIndex;
      SetFilePointer(fHData, E, nil, File_begin);
    end;
  //Save record
  WriteFile(fHData, LoadDataPtr^, fDataSize, W, nil);
  If PDataBuffer(ActiveBuffer)^.IsNew then
    begin
      PDataBuffer(ActiveBuffer)^.IsNew := False;
      //Build Index later. If I have child tables, build now
      NeedsIndex := True;
      fNeedsSubIndex := True;
      BuildIndex;
      For I := 0 to Children.Count - 1 do
        With TSimpleTable(Children[I]) do
          MasterChangedProc;
    end;
end;

procedure TSimpleTable.SetData(Field: TField; Buffer: Pointer);
begin
end;

Procedure TSimpleTable.SetMasterTable(AValue: TSimpleTable);
begin
  If not (fMasterTable = nil) then
    fMasterTable.DeleteChild(Self);
  fMasterTable := AValue;
  If not (fMasterTable = nil) then
    fMasterTable.AddChild(Self);
end;

procedure TSimpleTable.SetRecNo(Value: Integer);
begin
  If (Value > -1) and (Value < RecordCount - 1) then
    fRecNo := Value;
end;

Procedure TSimpleTable.SetSortField(AIndex: Integer);
begin
  fSortField := AIndex;
  Refresh;
end;


//---------------------------TSentenceTable

constructor TSentenceTable.Create(AOwner: TComponent);
begin
  Inherited Create(AOwner);
  fHSaved := CreateFile('Sentences.dat', Generic_Read OR Generic_Write,
      File_Share_Read OR File_Share_Write, nil, Open_Always, 0, 0);
  SenBlob := CreateFile('SentBlob.dat', Generic_Read OR Generic_Write,
      File_Share_Read OR File_Share_Write, nil, Open_Always, 0, 0);
  fDataSize := SizeOf(TSentence);
  LoadDataPtr := Addr(Data);
  AutoInc := @Data.ID;
  AutoIncSource := @DM.IDNumberList.SentenceID;
end;

Procedure TSentenceTable.DeleteAll;
begin
  //Also wipe the BLOB file
  SetFilePointer(SenBlob, 0, nil, File_begin);
  SetEndOfFile(SenBlob);
  Inherited DeleteAll;
end;

procedure TSentenceTable.GetData(Field: TField; Buffer: Pointer);
var
  Cs: PChar;
  W: DWord;
begin
  Case Field.Index of
    0: pInteger(Buffer)^ := Data.ID;
    1:
      begin
        //Get the sentence data. If Len > 80 then as BLOB
        If Data.Len <= 80 then
          StrPCopy(Buffer, Data.Text)
        else
          begin
            //Point to it
            SetFilePointer(SenBlob, Data.FLow, @Data.FHigh, File_Begin);
            //Load the new text data from file
            GetMem(Cs, Data.Len + 1);
            FillChar(Cs^, Data.Len + 1, 0);
            try
              ReadFile(SenBlob, Cs^, Data.Len, W, nil);
              StrCopy(Buffer, Cs);
            finally
              FreeMem(Cs, Data.Len + 1);
            end;
          end;
      end;
    2:
      begin
        //Refer to parent Text Object
        //StrPCopy(Buffer, Tx.URL);
      end;
    3: StrPCopy(Buffer, 'None');
    4: pInteger(Buffer)^ := Data.HitsLatest;
    5: pInteger(Buffer)^ := Data.HitsBefore;
    6: pInteger(Buffer)^ := Data.HitsLatest - Data.HitsBefore;
  end;
end;

procedure TSentenceTable.InternalInitFieldDefs;
begin
  FieldDefs.Clear;
  TFieldDef.Create(FieldDefs, 'ID', ftInteger, 0, False, 0);
  TFieldDef.Create(FieldDefs, 'Text', ftString, 4096, False, 1);
  TFieldDef.Create(FieldDefs, 'URL', ftString, 256, False, 2);
  TFieldDef.Create(FieldDefs, 'Category', ftString, 256, False, 3);
  TFieldDef.Create(FieldDefs, 'HitsLatest', ftInteger, 0, False, 4);
  TFieldDef.Create(FieldDefs, 'HitsBefore', ftInteger, 0, False, 5);
  TFieldDef.Create(FieldDefs, 'HitsNew', ftInteger, 0, False, 6);
end;

procedure TSentenceTable.SetData(Field: TField; Buffer: Pointer);
var
  W: DWord;
  A: String;
  Cs: PChar;
begin
  Case Field.Index of
    1:
      begin
        A := StrPas(Buffer);
        Data.Len := Length(A);
        If Data.Len <= 80 then
          Data.Text := A
        else
          begin
            //BLOB it
            Data.Text := '';
            If Data.Len >= Data.AllocLen then
              begin
                //Re-Allocate to end of File
                Data.FLow := GetFileSize(SenBlob, @Data.FHigh);
                Data.AllocLen := Data.Len;  //Allocate exact
              end;
            SetFilePointer(SenBLob, Data.FLow, @Data.FHigh, File_Begin);
            Cs := PChar(A);
            //Write the data
            WriteFile(SenBlob, Cs^, Data.Len, W, nil);
            //Write 16 at the end for Alloc
            //If ReAlloc then
            //   FillChar(EmptyBuffer, 16, 0);
            //   WriteFile(DataFile, EmptyBuffer, 16, W, nil);
          end;
      end;
    //3: Ignored
    4: Data.HitsLatest := pInteger(Buffer)^;
    5: Data.HitsBefore := pInteger(Buffer)^;
  end;
  //If PDataBuffer(ActiveBuffer)^.IsNew then
  //  Data.TextID := (Fetch the ID from Wherever ID are distributed
end;

Initialization
  DM := TDM.Create;
Finalization
  DM.Free;
end.
